use core:lang;
use lang:bs;
use lang:bs:macro;
use layout;

/**
 * Functions used by the syntax.
 */


/**
 * A presentation declaration.
 */
package class PresDecl extends NamedDecl {
	SStr name;
	Str title;
	Scope scope;
	SPresCont content;

	init(SStr name, Str title, Scope scope, SPresCont content) {
		init() {
			name = name;
			title = title;
			scope = scope;
			content = content;
		}
	}

	// Create a named entity.
	Named doCreate() {
		PresFn(name, title, scope, content);
	}

	// Update!
	Named? update(Scope scope) {
		var found = scope.find(Name(name.v));
		if (found as PresFn) {
			found.update(title, scope, content);
		} else if (found.empty) {
			var c = create();
			resolve();
			return c;
		}

		return null;
	}
}


/**
 * A function that declares a presentation.
 */
package class PresFn extends BSRawFn {
	Str title;
	Scope scope;
	SPresCont content;

	init(SStr name, Str title, Scope scope, SPresCont content) {
		init(Value(named{Presentation}), name, ValParam[], named{ui:Render}) {
			title = title;
			scope = scope;
			content = content;
		}
	}

	// Update ourselves.
	void update(Str title, Scope scope, SPresCont content) {
		this.title = title;
		this.scope = scope;
		this.content = content;
		reset();
	}

	// Create the function's body.
	FnBody createBody() {
		FnBody body(this, scope);

		// Create a presentation in a variable named 'this'.
		Var thisVar(body, Value(named{Presentation}), SStr("this"), Actuals(strConstant(SStr(title))));
		body.add(thisVar);

		content.transform(body);

		// Return the presentation.
		body.add(LocalVarAccess(SrcPos(), thisVar.var));

		// print(body.toS);
		body;
	}
}


// Apply the slide layout. Also: jam it inside a SlideBorder layout.
Expr slideLayout(Block inside, LayoutRoot layout, SStr? name, Expr? intro) on Compiler {
	var create = if (intro) {
		pattern (inside) {
			presentation:SlideBorder layout(this);
			layout.add(layout:Component:toAdd(${layout.block}));

			presentation:Slide slide(layout);
			slide.intro = ${intro};
			this.add(slide);
			slide;
		};
	} else {
		pattern (inside) {
			presentation:SlideBorder layout(this);
			layout.add(layout:Component:toAdd(${layout.block}));

			presentation:Slide slide(layout);
			this.add(slide);
			slide;
		};
	};

	if (name) {
		// If we have a name, we want to create a variable!
		Var(inside, name, create);
	} else {
		// Otherwise, we just execute the statements.
		create;
	}
}

// Apply the slide layout, but don't add a border.
Expr slidePlainLayout(Block inside, LayoutRoot layout, SStr? name, Expr? intro) on Compiler {
	var create = if (intro) {
		pattern (inside) {
			presentation:Slide slide(${layout.block});
			slide.intro = ${intro};
			this.add(slide);
			slide;
		};
	} else {
		pattern (inside) {
			presentation:Slide slide(${layout.block});
			this.add(slide);
			slide;
		};
	};

	if (name) {
		Var(inside, name, create);
	} else {
		create;
	}
}

// Set the background. Does not use a SlideBorder layout.
Expr slideBackground(Block inside, LayoutRoot layout) on Compiler {
	pattern(inside) {
		this.background(presentation:Slide(${layout.block}));
	};
}

// Get a slide intro expression.
Expr slideIntro(Block block, SrcName name, Actuals params) on Compiler {
	name.last.name = name.last.name + "Intro";
	namedExpr(block, name, params);
}

// Animation declaration.
class AniDecl on Compiler {
	// Block where we store everything.
	ExprBlock block;

	// Variable accessing the created type.
	LocalVarAccess created;

	init(Block parent, Str step, SrcName type, Actuals params) {
		ExprBlock b(type.pos, parent);
		type.last.name = type.last.name + "Animation";
		params.addFirst(intConstant(type.pos, step));
		Var ani(b, SStr("@ animation", type.pos), namedExpr(b, type, params));
		b.add(ani);

		init() {
			block = b;
			created(type.pos, ani.var);
		}
	}

	// Finalize this block.
	void finalize() {
		block.add(created);
	}

	// Set a property.
	void setProperty(Str property, Expr to) {
		var lhs = namedExpr(block, SStr(property, to.pos), created, Actuals());

		AssignOpInfo assign(SStr("=", to.pos), 10, true);
		block.add(assign.meaning(block, lhs, to));
	}

	void setDuration(Expr to) {
		setProperty("duration", to);
	}

	void setOffset(Expr to) {
		setProperty("offset", to);
	}
}

void add(LayoutBlock block, AniDecl ani) on Compiler {
	ani.finalize();
	block.add(SStr("animation", ani.block.pos), Actuals(ani.block));
}
