/* This file is part of the KDE project
    SPDX-FileCopyrightText: 1998, 1999, 2010 David Faure <faure@kde.org>

    SPDX-License-Identifier: GPL-2.0-or-later
*/
#include "konqmisc.h"
#include "konqsettings.h"
#include "konqmainwindow.h"
#include "konqviewmanager.h"
#include "konqview.h"
#include "konqmainwindowfactory.h"
#include "konqurl.h"

#include "libkonq_utils.h"

#include "konqdebug.h"
#include <kurifilter.h>
#include <KLocalizedString>

#include <kprotocolmanager.h>
#include <kiconloader.h>
#include <kconfiggroup.h>
#include <QList>
#include <QStandardPaths>

/**********************************************
 *
 * KonqMisc
 *
 **********************************************/
KonqMainWindow *KonqMisc::newWindowFromHistory(KonqView *view, int steps)
{
    int oldPos = view->historyIndex();
    int newPos = oldPos + steps;

    const HistoryEntry *he = view->historyAt(newPos);
    if (!he) {
        return nullptr;
    }

    KonqMainWindow *mainwindow = KonqMainWindowFactory::createEmptyWindow();
    if (!mainwindow) {
        return nullptr;
    }
    KonqView *newView = mainwindow->currentView();

    if (!newView) {
        return nullptr;
    }

    newView->copyHistory(view);
    newView->setHistoryIndex(newPos);
    newView->restoreHistory();
    mainwindow->show();
    return mainwindow;
}

QUrl KonqMisc::konqFilteredURL(KonqMainWindow *parent, const QString &_url, const QUrl &currentDirectory)
{
    Q_UNUSED(parent); // Useful if we want to change the error handling again

    QUrl url(_url);

    //Don't filter konq: or data: URLs as they aren't known to KUriFilter and it would
    //reject them
    if (KonqUrl::canBeKonqUrl(_url)) {
        const bool isKnownAbout = KonqUrl::hasKnownPathRoot(_url);
        return isKnownAbout ? QUrl(_url) : KonqUrl::url(KonqUrl::Type::NoPath);
    } else if (url.scheme() == QLatin1String("data")) {
        return url;
    }

    KUriFilterData data(_url);
    if (currentDirectory.isLocalFile()) {
        data.setAbsolutePath(currentDirectory.toLocalFile());
    }
    // We do not want to the filter to check for executables
    // from the location bar.
    data.setCheckForExecutables(false);
    bool filtered = KUriFilter::self()->filterUri(data);
    return urlFromURIFilterResult(filtered, data);
}

QUrl KonqMisc::urlFromURIFilterResult(bool filterSuccess, const KUriFilterData& data)
{
    if (filterSuccess && data.uriType() != KUriFilterData::Error) {
        return data.uri();
    }
    KIO::Error error = KIO::ERR_MALFORMED_URL;
    QString origUrl = data.typedString();
    QString msg = origUrl;
    QUrl url = QUrl(origUrl);
    if (filterSuccess && !data.errorMsg().isEmpty()) {
        error = KIO::ERR_WORKER_DEFINED;
        msg = data.errorMsg();
    } else if (const QString scheme = data.uri().scheme(); !scheme.isEmpty() && !KProtocolInfo::isKnownProtocol(scheme)) {
        // Show an explicit error for an unknown URL scheme, because the
        // explanation generated by KIO::rawErrorDetail() gives useful
        // information.
        error = KIO::ERR_UNSUPPORTED_PROTOCOL;
    }
    // NOTE: a valid URL like http://kde.org always passes the filtering test.
    // As such, this point could only be reached when origUrl is NOT a valid URL.
    return Konq::makeErrorUrl(error, origUrl, url);
}

QString KonqMisc::encodeFilename(QString filename)
{
    return filename.replace(':', '_');
}

QString KonqMisc::decodeFilename(QString filename)
{
    return filename.replace('_', ':');
}

