# Copyright 1999-2025 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

FORTRAN_NEEDED=fortran
DISTUTILS_EXT=1
DISTUTILS_USE_PEP517=meson-python
PYTHON_COMPAT=( pypy3_11 python3_{11..13} )
PYTHON_REQ_USE="threads(+)"

inherit flag-o-matic fortran-2 distutils-r1

DESCRIPTION="Scientific algorithms library for Python"
HOMEPAGE="
	https://scipy.org/
	https://github.com/scipy/scipy/
	https://pypi.org/project/scipy/
"

if [[ ${PV} == *9999* ]] ; then
	inherit git-r3

	# Need submodules, so git for now.
	EGIT_REPO_URI="https://github.com/scipy/scipy"
	EGIT_BRANCH="maintenance/$(ver_cut 1-2).x"
	EGIT_SUBMODULES=( '*' )
else
	inherit pypi

	# Upstream is often behind with doc updates
	DOC_PV=${PV}

	SRC_URI+="
		doc? (
			https://docs.scipy.org/doc/${PN}-${DOC_PV}/${PN}-html-${DOC_PV}.zip
		)
	"

	if [[ ${PV} != *rc* ]] ; then
		KEYWORDS="amd64 arm arm64 ~loong ~ppc ppc64 ~riscv ~s390 ~sparc x86"
	fi
fi

LICENSE="BSD LGPL-2"
SLOT="0"
IUSE="doc +fortran test-rust"

# pythran is needed in both BDEPEND (for /usr/bin/pythran) and in DEPEND
# to actually compile code generated by pythran (headers and such needed).
#
# umfpack is technically optional but it's preferred to have it available.
COMMON_DEPEND="
	>=dev-python/numpy-1.23.5:=[lapack,${PYTHON_USEDEP}]
	>=media-libs/qhull-2020.2:=
	virtual/cblas
	>=virtual/lapack-3.8
"
# Only boost.math is used, and meson.build doesn't even look up specific boost modules.
DEPEND="
	${COMMON_DEPEND}
	>=dev-libs/boost-1.88.0
"
RDEPEND="
	${COMMON_DEPEND}
	dev-python/pillow[${PYTHON_USEDEP}]
"
DEPEND+="
	fortran? (
		>=dev-python/pythran-0.16.0[${PYTHON_USEDEP}]
	)
"
BDEPEND="
	dev-lang/swig
	>=dev-python/cython-3.0.8[${PYTHON_USEDEP}]
	>=dev-python/meson-python-0.15.0[${PYTHON_USEDEP}]
	>=dev-python/pybind11-2.13.2[${PYTHON_USEDEP}]
	>=dev-python/pytest-8.0.0[${PYTHON_USEDEP}]
	>=dev-build/meson-1.5.0
	!kernel_Darwin? ( dev-util/patchelf )
	virtual/pkgconfig
	doc? ( app-arch/unzip )
	fortran? (
		>=dev-python/pythran-0.16.0[${PYTHON_USEDEP}]
	)
	test-rust? (
		dev-python/pooch[${PYTHON_USEDEP}]
	)
"

EPYTEST_PLUGINS=( hypothesis )
EPYTEST_XDIST=1
distutils_enable_tests pytest

src_unpack() {
	default

	if use doc; then
		unzip -qo "${DISTDIR}"/${PN}-html-${DOC_PV}.zip -d html || die
	fi
}

src_prepare() {
	distutils-r1_src_prepare

	# unpin system dependencies
	sed -i -e "s@version : '@&>=@" meson.build || die
}

python_configure_all() {
	DISTUTILS_ARGS=(
		-Dblas=blas
		-Dlapack=lapack
		-Duse-pythran=$(usex fortran true false)
		-Duse-system-libraries=all
	)

	# https://bugs.gentoo.org/932721
	has_version '>=dev-python/numpy-2.0.0' && filter-lto
}

python_test() {
	# We run tests in parallel, so avoid having n^2 threads in lapack
	# tests.
	local -x BLIS_NUM_THREADS=1
	local -x MKL_NUM_THREADS=1
	local -x OMP_NUM_THREADS=1
	local -x OPENBLAS_NUM_THREADS=1

	cd "${BUILD_DIR}/install$(python_get_sitedir)" || die

	local EPYTEST_DESELECT=(
		# Network
		scipy/datasets/tests/test_data.py::TestDatasets::test_existence_all
		scipy/datasets/tests/test_data.py::TestDatasets::test_ascent
		scipy/datasets/tests/test_data.py::TestDatasets::test_face
		scipy/datasets/tests/test_data.py::TestDatasets::test_electrocardiogram

		# Crashes with assertion, not a regression
		# https://github.com/scipy/scipy/issues/19321
		scipy/signal/tests/test_signaltools.py::test_lfilter_bad_object

		# Awfully slow tests
		scipy/interpolate/tests/test_bsplines.py::TestBatch::test_batch
		scipy/linalg/tests/test_basic.py::TestLstsq::test_random_complex_exact
		scipy/sparse/linalg/_eigen/tests/test_svds.py::test_small_sigma_sparse

		# TODO: minor precision errors
		'scipy/linalg/tests/test_batch.py::TestBatch::test_solve[float32-bdim2]'
		'scipy/linalg/tests/test_batch.py::TestBatch::test_lu_solve[float32-bdim2]'
		'scipy/stats/tests/test_continuous.py::TestDistributions::test_funcs[cdf-methods11-x-Normal]['

		# Crashes, probably too big
		'scipy/interpolate/tests/test_fitpack2.py::TestRectBivariateSpline::test_spline_large_2d_maxit'
	)
	local EPYTEST_IGNORE=()

	if ! has_version -b "dev-python/pooch[${PYTHON_USEDEP}]" ; then
		EPYTEST_IGNORE+=(
			scipy/datasets/tests/test_data.py
		)
	fi

	case ${EPYTHON} in
		pypy3*)
			EPYTEST_DESELECT+=(
				# TODO
				scipy/special/tests/test_data.py::test_boost
				scipy/_lib/tests/test_ccallback.py::test_callbacks
			)
			;;
	esac

	# avoid other stuff being multithreaded when using xdist
	local -x PYTHON_CPU_COUNT=1
	epytest scipy
}

python_install_all() {
	use doc && local HTML_DOCS=( "${WORKDIR}"/html/. )

	distutils-r1_python_install_all
}
