/*
* Crystals Dilithium Digital Signature Algorithms
* Based on the public domain reference implementation by the
* designers (https://github.com/pq-crystals/dilithium)
*
* Further changes
* (C) 2021-2023 Jack Lloyd
* (C) 2021-2022 Manuel Glaser - Rohde & Schwarz Cybersecurity
* (C) 2021-2023 Michael Boric, René Meusel - Rohde & Schwarz Cybersecurity
*
* Botan is released under the Simplified BSD License (see license.txt)
*/

#ifndef BOTAN_DILITHIUM_COMMON_H_
#define BOTAN_DILITHIUM_COMMON_H_

#include <botan/pk_keys.h>

namespace Botan {

class BOTAN_PUBLIC_API(3, 0) DilithiumMode {
   public:
      enum Mode : uint8_t {
         Dilithium4x4 = 1,
         Dilithium4x4_AES BOTAN_DEPRECATED("Dilithium AES mode is deprecated"),
         Dilithium6x5,
         Dilithium6x5_AES BOTAN_DEPRECATED("Dilithium AES mode is deprecated"),
         Dilithium8x7,
         Dilithium8x7_AES BOTAN_DEPRECATED("Dilithium AES mode is deprecated"),
         ML_DSA_4x4,
         ML_DSA_6x5,
         ML_DSA_8x7,
      };

   public:
      // NOLINTNEXTLINE(*-explicit-conversions)
      DilithiumMode(Mode mode) : m_mode(mode) {}

      explicit DilithiumMode(const OID& oid);
      explicit DilithiumMode(std::string_view str);

      OID object_identifier() const;
      std::string to_string() const;

      BOTAN_DEPRECATED("Dilithium AES mode is deprecated") bool is_aes() const;
      BOTAN_DEPRECATED("Dilithium AES mode is deprecated") bool is_modern() const;
      bool is_ml_dsa() const;

      bool is_dilithium_round3() const { return !is_ml_dsa(); }

      bool is_available() const;

      Mode mode() const { return m_mode; }

   private:
      Mode m_mode;
};

class Dilithium_PublicKeyInternal;
class Dilithium_PrivateKeyInternal;

/**
 * This implementation is based on
 * https://github.com/pq-crystals/dilithium/commit/3e9b9f1412f6c7435dbeb4e10692ea58f181ee51
 *
 * Note that this is _not_ compatible with the round 3 submission of the NIST competition.
 */
class BOTAN_PUBLIC_API(3, 0) Dilithium_PublicKey : public virtual Public_Key {
   public:
      std::string algo_name() const override;

      AlgorithmIdentifier algorithm_identifier() const override;

      OID object_identifier() const override;

      size_t key_length() const override;

      size_t estimated_strength() const override;

      std::vector<uint8_t> raw_public_key_bits() const override;

      std::vector<uint8_t> public_key_bits() const override;

      bool check_key(RandomNumberGenerator& rng, bool strong) const override;

      bool supports_operation(PublicKeyOperation op) const override { return (op == PublicKeyOperation::Signature); }

      std::unique_ptr<Private_Key> generate_another(RandomNumberGenerator& rng) const final;

      Dilithium_PublicKey(const AlgorithmIdentifier& alg_id, std::span<const uint8_t> pk);

      Dilithium_PublicKey(std::span<const uint8_t> pk, DilithiumMode mode);

      std::unique_ptr<PK_Ops::Verification> create_verification_op(std::string_view params,
                                                                   std::string_view provider) const override;

      std::unique_ptr<PK_Ops::Verification> create_x509_verification_op(const AlgorithmIdentifier& signature_algorithm,
                                                                        std::string_view provider) const override;

   protected:
      Dilithium_PublicKey() = default;

      friend class Dilithium_Verification_Operation;
      friend class Dilithium_Signature_Operation;

      std::shared_ptr<Dilithium_PublicKeyInternal> m_public;  // NOLINT(*non-private-member-variable*)
};

BOTAN_DIAGNOSTIC_PUSH
BOTAN_DIAGNOSTIC_IGNORE_INHERITED_VIA_DOMINANCE

class BOTAN_PUBLIC_API(3, 0) Dilithium_PrivateKey final : public virtual Dilithium_PublicKey,
                                                          public virtual Botan::Private_Key {
   public:
      std::unique_ptr<Public_Key> public_key() const override;

      /**
       * Generates a new key pair
       */
      Dilithium_PrivateKey(RandomNumberGenerator& rng, DilithiumMode mode);

      /**
       * Read an encoded private key.
       */
      Dilithium_PrivateKey(const AlgorithmIdentifier& alg_id, std::span<const uint8_t> sk);

      /**
       * Read an encoded private key given the dilithium @p mode.
       */
      Dilithium_PrivateKey(std::span<const uint8_t> sk, DilithiumMode mode);

      secure_vector<uint8_t> private_key_bits() const override;

      secure_vector<uint8_t> raw_private_key_bits() const override;

      /**
       * Create a signature operation that produces a Dilithium signature either
       * with "Randomized" or "Deterministic" rhoprime. Pass either of those
       * strings as @p params. Default (i.e. empty @p params is "Randomized").
       */
      std::unique_ptr<PK_Ops::Signature> create_signature_op(RandomNumberGenerator& rng,
                                                             std::string_view params,
                                                             std::string_view provider) const override;

   private:
      friend class Dilithium_Signature_Operation;

      std::shared_ptr<Dilithium_PrivateKeyInternal> m_private;
};

BOTAN_DIAGNOSTIC_POP

}  // namespace Botan

#endif
