;;; tex-site.el - Site specific variables.  Don't edit.  -*- lexical-binding: t; -*-

;; Copyright (C) 2005-2024  Free Software Foundation, Inc.
;;
;; completely rewritten.

;; Author: David Kastrup <dak@gnu.org>
;; Maintainer: auctex-devel@gnu.org
;; Keywords: tex

;; This file is part of AUCTeX.

;; AUCTeX is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; AUCTeX is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with AUCTeX; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
;; 02110-1301, USA.

;;; Commentary:

;; This file contains startup code, autoloads and variables adapted to
;; the local site configuration.  It is generated and placed by the
;; installation procedure and should not be edited by hand, nor moved
;; to a different place, as some settings may be established relative
;; to the file.

;; All user customization should be done with
;; M-x customize-variable RET

;;; Code:

(when (< emacs-major-version 27)
  (error "AUCTeX requires Emacs 27.1 or later"))

(declare-function BibTeX-auto-store "latex")

(unless (or (fboundp 'TeX-modes-set)     ;Avoid inf-looping.
            (fboundp 'TeX-tex-mode))     ;auctex-autoloads is not loaded.
  ;; Try and support the case where someone loads tex-site.el or
  ;; auctex.el directly, in the old way.
  (provide 'tex-site)        ;Avoid (re)loading tex-site from auctex-autoloads.

  (load "auctex-autoloads" 'noerror 'nomessage))

;; Define here in order for `M-x customize-group <RET> AUCTeX <RET>'
;; to work if the main AUCTeX files are not loaded yet.
(defgroup AUCTeX nil
  "A (La)TeX environment."
  :tag "AUCTeX"
  :link '(custom-manual "(auctex)Top")
  :link '(url-link :tag "Home Page" "https://www.gnu.org/software/auctex/")
  :prefix "TeX-"
  :group 'tex
  :load "tex" :load "latex" :load "tex-style")

(defvar TeX-lisp-directory
  (directory-file-name (file-name-directory load-file-name))
  "The directory where most of the AUCTeX lisp files are located.
For the location of lisp files associated with
styles, see the variables TeX-style-* (hand-generated lisp) and
TeX-auto-* (automatically generated lisp).")

(add-to-list 'load-path TeX-lisp-directory)

(defvar TeX-data-directory
  "/usr/share/emacs/etc/auctex"
  "The directory where the AUCTeX non-Lisp data is located.")

(defcustom TeX-auto-global
  "/var/lib/auctex"
  "Directory containing automatically generated information.

For storing automatic extracted information about the TeX macros
shared by all users of a site."
  :group 'TeX-file
  :type 'directory)

(defconst TeX-mode-alist
  '((tex-mode . TeX-tex-mode)
    (plain-tex-mode . plain-TeX-mode)
    (texinfo-mode . Texinfo-mode)
    (latex-mode . LaTeX-mode)
    (doctex-mode . docTeX-mode))
  "Alist of built-in TeX modes and their counterparts in AUCTeX.")

(defalias 'TeX-load-hack #'ignore)

(defun tex-site-unload-function ()
  (TeX-modes-set 'TeX-modes nil)

  ;; COMPATIBILITY for Emacs<29
  (put 'plain-TeX-mode 'auctex-function-definition nil)
  (put 'LaTeX-mode 'auctex-function-definition nil)
  (put 'TeX-mode 'auctex-function-definition nil)

  (setq load-path (delq TeX-lisp-directory load-path))
  ;; Tell emacs to continue standard unloading procedure.
  nil)

(defun TeX-modes-set (var value &optional _ignored)
  "Set VAR (which should be `TeX-modes') to VALUE.

Arrange the redirection of the built-in TeX modes according to VALUE.
- The built-in modes in VALUE are redirected to the corresponding
  AUCTeX major modes.
- The built-in modes not in VALUE discard redirection, if any.
If `major-mode-remap-alist' is available, use it for redirection.
Otherwise, use advice facility."
  (custom-set-default var value)
  (let (elt dst)
    (dolist (entry TeX-mode-alist)
      (setq elt (car entry)
            dst (cdr entry))
      (if (memq elt value)
          (if (boundp 'major-mode-remap-alist)
              (or (eq (cdr-safe (assq elt major-mode-remap-alist)) dst)
                  (push (cons elt dst) major-mode-remap-alist))
            ;; COMPATIBILITY for Emacs<29
            (advice-add elt :override dst
                        ;; COMPATIBILITY for Emacs 28.[12]
                        ;; Give it higher precedence than the :around
                        ;; advice given to `tex-mode' in tex-mode.el.
                        ;; <URL:https://lists.gnu.org/r/auctex-devel/2022-09/msg00050.html>
                        '((depth . -10))))
        (if (boundp 'major-mode-remap-alist)
            (setq major-mode-remap-alist
                  (delete entry major-mode-remap-alist))
          ;; COMPATIBILITY for Emacs<29
          (advice-remove elt dst))))))

(defcustom TeX-modes
  (mapcar #'car TeX-mode-alist)
  "List of built-in TeX modes redirected to AUCTeX modes.

This variable can't be set normally; use customize for that, or
set it with `TeX-modes-set'."
  :type (cons 'set
              (mapcar (lambda(x) (list 'const (car x))) TeX-mode-alist))
  :set #'TeX-modes-set
  :initialize #'custom-initialize-reset)

(defun TeX--alias-overlapped-modes (&optional restore)
  "Delete or restore definition of overlapped modes via `defalias'.
Set function definition for modes overlapped between tex-mode.el
and AUCTeX, `plain-TeX-mode', `LaTeX-mode' and `TeX-mode'.
If optional argument RESTORE is nil, delete the definition.
Otherwise, restore AUCTeX definition saved in the symbol property
`auctex-function-definition'."
  (dolist (mode '(plain-TeX-mode LaTeX-mode TeX-mode))
    (if (eq (symbol-function mode)
            (intern (downcase (symbol-name mode))))
        (defalias mode (if restore
                           (get mode 'auctex-function-definition))))))

;; COMPATIBILITY for Emacs<29, which executes
;; (defalias 'LaTeX-mode #'latex-mode) etc. in tex-mode.el.
(with-eval-after-load 'tex-mode
  ;; This must be no-op after (unload-feature 'tex-site).
  (if (featurep 'tex-site)
      (TeX--alias-overlapped-modes t)))

(defconst AUCTeX-version "13.3"
  "AUCTeX version.
If not a regular release, the date of the last change.")

(defconst AUCTeX-date "2024-01-17"
  "AUCTeX release date using the ISO 8601 format, yyyy-mm-dd.")

;; Store bibitems when saving a BibTeX buffer
(add-hook 'bibtex-mode-hook #'BibTeX-auto-store)

;;; auto-loads.el --- automatically extracted autoloads (do not edit)
;; Generated by the `loaddefs-generate' function.

;; This file is part of GNU Emacs.

;;; Code:



;;; Generated autoloads from bib-cite.el

(autoload 'bib-cite-minor-mode "bib-cite" "\
Toggle bib-cite mode.
When bib-cite mode is enabled, citations, labels and refs are highlighted
when the mouse is over them.  Clicking on these highlights with [mouse-2]
runs `bib-find', and [mouse-3] runs `bib-display'.

(fn ARG)" t)
(autoload 'turn-on-bib-cite "bib-cite" "\
Unconditionally turn on Bib Cite mode.")
(register-definition-prefixes "bib-cite" '("LaTeX-find-label-hist-alist" "bib-" "create-alist-from-list" "member-cis" "psg-" "search-directory-tree"))


;;; Generated autoloads from context.el

(defalias 'context-mode #'ConTeXt-mode)
(autoload 'ConTeXt-mode "context" "\
Major mode in AUCTeX for editing ConTeXt files.

Entering `ConTeXt-mode' calls the value of `text-mode-hook',
then the value of `TeX-mode-hook', and then the value
of `ConTeXt-mode-hook'.

(fn)" t)
(register-definition-prefixes "context" '("ConTeXt-" "TeX-ConTeXt-sentinel" "context-guess-current-interface"))


;;; Generated autoloads from context-en.el

(register-definition-prefixes "context-en" '("ConTeXt-"))


;;; Generated autoloads from context-nl.el

(register-definition-prefixes "context-nl" '("ConTeXt-"))


;;; Generated autoloads from font-latex.el

(autoload 'font-latex-setup "font-latex" "\
Setup this buffer for LaTeX font-lock.  Usually called from a hook.")
(register-definition-prefixes "font-latex" '("font-latex-"))


;;; Generated autoloads from latex.el

(autoload 'BibTeX-auto-store "latex" "\
This function should be called from `bibtex-mode-hook'.
It will setup BibTeX to store keys in an auto file.")
(add-to-list 'auto-mode-alist '("\\.drv\\'" . LaTeX-mode) t)
(add-to-list 'auto-mode-alist '("\\.hva\\'" . LaTeX-mode))
 (if (eq (symbol-function 'LaTeX-mode) 'latex-mode)
    (defalias 'LaTeX-mode nil))
(autoload 'LaTeX-mode "latex" "\
Major mode in AUCTeX for editing LaTeX files.
See info under AUCTeX for full documentation.

Entering LaTeX mode calls the value of `text-mode-hook',
then the value of `TeX-mode-hook', and then the value
of `LaTeX-mode-hook'.

(fn)" t)
(put 'LaTeX-mode 'auctex-function-definition (symbol-function 'LaTeX-mode))
(autoload 'docTeX-mode "latex" "\
Major mode in AUCTeX for editing .dtx files derived from `LaTeX-mode'.
Runs `LaTeX-mode', sets a few variables and
runs the hooks in `docTeX-mode-hook'.

(fn)" t)
(register-definition-prefixes "latex" '("Bib" "LaTeX-" "TeX-" "docTeX-" "latex-math-mode"))


;;; Generated autoloads from latex-flymake.el

(register-definition-prefixes "latex-flymake" '("LaTeX-"))


;;; Generated autoloads from multi-prompt.el

(autoload 'multi-prompt "multi-prompt" "\
Completing prompt for a list of strings.
The first argument SEPARATOR should be the string (of length 1) to
separate the elements in the list.  The second argument UNIQUE should
be non-nil, if each element must be unique.  The remaining elements
are the arguments to `completing-read'.  See that.

(fn SEPARATOR UNIQUE PROMPT TABLE &optional MP-PREDICATE REQUIRE-MATCH INITIAL HISTORY)")
(autoload 'multi-prompt-key-value "multi-prompt" "\
Read multiple strings, with completion and key=value support.
PROMPT is a string to prompt with, usually ending with a colon
and a space.

TABLE is an alist where each entry is a list.  The first element
of each list is a string representing a key and the optional
second element is a list with strings to be used as values for
the key.  The second element can also be a variable returning a
list of strings.

See the documentation for `completing-read' for details on the
other arguments: PREDICATE, REQUIRE-MATCH, INITIAL-INPUT, HIST,
DEF, and INHERIT-INPUT-METHOD.

The return value is the string as entered in the minibuffer.

(fn PROMPT TABLE &optional PREDICATE REQUIRE-MATCH INITIAL-INPUT HIST DEF INHERIT-INPUT-METHOD)")
(register-definition-prefixes "multi-prompt" '("multi-prompt-"))


;;; Generated autoloads from plain-tex.el

 (if (eq (symbol-function 'plain-TeX-mode) 'plain-tex-mode)
    (defalias 'plain-TeX-mode nil))
(autoload 'plain-TeX-mode "plain-tex" "\
Major mode in AUCTeX for editing plain TeX files.
See info under AUCTeX for documentation.

Entering `plain-TeX-mode' calls the value of `text-mode-hook',
then the value of `TeX-mode-hook', and then the value
of `plain-TeX-mode-hook'.

(fn)" t)
(put 'plain-TeX-mode 'auctex-function-definition (symbol-function 'plain-TeX-mode))
(autoload 'AmSTeX-mode "plain-tex" "\
Major mode in AUCTeX for editing AmSTeX files.
See info under AUCTeX for documentation.

Entering `AmSTeX-mode' calls the value of `text-mode-hook', then
the value of `TeX-mode-hook', `plain-TeX-mode-hook' and then the
value of `AmSTeX-mode-hook'.

(fn)" t)
(defalias 'ams-tex-mode #'AmSTeX-mode)
(register-definition-prefixes "plain-tex" '("AmSTeX-" "plain-TeX-"))


;;; Generated autoloads from tex.el

(autoload 'TeX-tex-mode "tex" "\
Call suitable AUCTeX major mode for editing TeX or LaTeX files.
Tries to guess whether this file is for plain TeX or LaTeX.

The algorithm is as follows:

   1) If the file is empty or `TeX-force-default-mode' is not set to nil,
      `TeX-default-mode' is chosen.
   2) If non-commented out content matches with regular expression in
      `TeX-format-list', use the associated major mode.  For example,
      if \\documentclass or \\begin{, \\section{, \\part{ or \\chapter{ is
      found, `LaTeX-mode' is selected.
   3) Otherwise, use `TeX-default-mode'.

By default, `TeX-format-list' has a fallback entry for
`plain-TeX-mode', thus non-empty file which didn't match any
other entries will enter `plain-TeX-mode'." t)
 (if (eq (symbol-function 'TeX-mode) 'tex-mode)
    (defalias 'TeX-mode nil))
(put 'TeX-mode 'auctex-function-definition (symbol-function 'TeX-mode))
(autoload 'TeX-auto-generate "tex" "\
Generate style file for TEX and store it in AUTO.
If TEX is a directory, generate style files for all files in the directory.

(fn TEX AUTO)" t)
(autoload 'TeX-auto-generate-global "tex" "\
Create global auto directory for global TeX macro definitions." t)
(autoload 'TeX-submit-bug-report "tex" "\
Submit a bug report on AUCTeX via mail.

Don't hesitate to report any problems or inaccurate documentation.

If you don't have setup sending mail from Emacs, please copy the
output buffer into your mail program, as it gives us important
information about your AUCTeX version and AUCTeX configuration." t)
(register-definition-prefixes "tex" '("Bib" "ConTeXt-" "LaTeX-" "TeX-" "docTeX-default-extension" "plain-TeX-auto-regexp-list" "tex-"))


;;; Generated autoloads from tex-bar.el

(autoload 'TeX-install-toolbar "tex-bar" "\
Install toolbar buttons for TeX mode." t)
(autoload 'LaTeX-install-toolbar "tex-bar" "\
Install toolbar buttons for LaTeX mode." t)
(register-definition-prefixes "tex-bar" '("TeX-bar-"))


;;; Generated autoloads from tex-fold.el

(autoload 'TeX-fold-mode "tex-fold" "\
Minor mode for hiding and revealing macros and environments.

Called interactively, with no prefix argument, toggle the mode.
With universal prefix ARG (or if ARG is nil) turn mode on.
With zero or negative ARG turn mode off.

(fn &optional ARG)" t)
(defalias 'tex-fold-mode #'TeX-fold-mode)
(register-definition-prefixes "tex-fold" '("TeX-fold-"))


;;; Generated autoloads from tex-font.el

(autoload 'tex-font-setup "tex-font" "\
Setup font lock support for TeX.")
(register-definition-prefixes "tex-font" '("tex-"))


;;; Generated autoloads from tex-info.el

(autoload 'Texinfo-mode "tex-info" "\
Major mode in AUCTeX for editing Texinfo files.

Entering Texinfo mode calls the value of `text-mode-hook' and then the
value of `Texinfo-mode-hook'.

(fn)" t)
(register-definition-prefixes "tex-info" '("Texinfo-" "texinfo-environment-regexp"))


;;; Generated autoloads from tex-ispell.el

(register-definition-prefixes "tex-ispell" '("TeX-ispell-"))


;;; Generated autoloads from tex-jp.el

(autoload 'japanese-plain-TeX-mode "tex-jp" "\
Major mode in AUCTeX for editing Japanese plain TeX files.

(fn)" t)
(defalias 'japanese-plain-tex-mode #'japanese-plain-TeX-mode)
(autoload 'japanese-LaTeX-mode "tex-jp" "\
Major mode in AUCTeX for editing Japanese LaTeX files.

(fn)" t)
(defalias 'japanese-latex-mode #'japanese-LaTeX-mode)
(register-definition-prefixes "tex-jp" '("TeX-japanese-process-" "japanese-"))


;;; Generated autoloads from tex-style.el

(register-definition-prefixes "tex-style" '("LaTeX-" "TeX-TikZ-point-name-regexp"))


;;; Generated autoloads from texmathp.el

(autoload 'texmathp "texmathp" "\
Determine if point is inside (La)TeX math mode.
Returns t or nil.  Additional info is placed into `texmathp-why'.
The functions assumes that you have (almost) syntactically correct (La)TeX in
the buffer.
See the variable `texmathp-tex-commands' about which commands are checked." t)
(autoload 'texmathp-match-switch "texmathp" "\
Search backward for any of the math switches.
Limit searched to BOUND.

(fn BOUND)")
(register-definition-prefixes "texmathp" '("texmathp-"))


;;; Generated autoloads from toolbar-x.el

 (autoload 'toolbarx-install-toolbar "toolbar-x")
(register-definition-prefixes "toolbar-x" '("toolbarx-"))

;;; End of scraped data


;; Local Variables:
;; version-control: never
;; no-byte-compile: t
;; no-update-autoloads: t
;; no-native-compile: t
;; coding: utf-8-emacs-unix
;; End:

(provide 'tex-site)
;;; tex-site.el ends here
