// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

#include "qqmlextensionplugin.h"
#include "qqmlextensionplugin_p.h"

QT_BEGIN_NAMESPACE

/*!
    \since 5.0
    \inmodule QtQml
    \class QQmlExtensionPlugin
    \brief The QQmlExtensionPlugin class provides an abstract base for custom QML extension plugins
           with custom type registration functions.

    \ingroup plugins

    \note If you need to write a plugin manually (which is rare) you should always use
    \l{QQmlEngineExtensionPlugin}. QQmlExtensionPlugin only provides the registerTypes() and
    unregisterTypes() functions in addition. You should not use them, but rather declare your
    types with \l{QML_ELEMENT} and friends and have the build system take care of the registration.
*/

/*!
    \since 5.14
    \inmodule QtQml
    \class QQmlEngineExtensionPlugin
    \brief The QQmlEngineExtensionPlugin class provides an abstract base for custom QML extension
           plugins.

    \ingroup plugins

    \include qqmlextensionplugin.qdocinc

    The \l {Writing QML Extensions with C++} tutorial also contains a chapter
    on creating QML plugins.

    \sa {How to Create Qt Plugins}
*/

/*!
    \fn void QQmlExtensionPlugin::registerTypes(const char *uri)

    Registers the QML types in the given \a uri. Subclasses should implement
    this to call \l qmlRegisterType() for all types which are
    provided by the extension plugin.

    The \a uri is an identifier for the plugin generated by the QML engine
    based on the name and path of the extension's plugin library.
*/

/*!
    \internal
*/
QQmlExtensionPlugin::QQmlExtensionPlugin(QObject *parent)
#if QT_DEPRECATED_SINCE(6, 3)
    : QObject(*(new QQmlExtensionPluginPrivate), parent)
#else
    : QObject(parent)
#endif
{
}

/*!
    Constructs a QML extension plugin with the given \a parent.

    Note that this constructor is invoked automatically by the
    Q_PLUGIN_METADATA() macro, so there is no need for calling it
    explicitly.
 */
QQmlEngineExtensionPlugin::QQmlEngineExtensionPlugin(QObject *parent)
    : QObject(parent)
{
}


/*!
  \internal
 */
QQmlExtensionPlugin::~QQmlExtensionPlugin() = default;

/*!
  \internal
 */
QQmlEngineExtensionPlugin::~QQmlEngineExtensionPlugin() = default;

#if QT_DEPRECATED_SINCE(6, 3)
/*!
    \since 5.1
    \internal
    \deprecated [6.3] This is unnecessary and doesn't work for optional plugins
    \brief Returns the URL of the directory from which the extension is loaded.

    This is useful when the plugin also needs to load QML files or other
    assets from the same directory.

    \note You should not need this function. Other files that are part of the
          module's public interface should be specified accordingly in the build
          system and qmldir file. The build system makes sure that they end up
          both in the final module directory, and in the resource file system.
          You can use the copy from the resource file system in the plugin.
          Non-QML/JS files private to the plugin can be added to the resource
          file system manually. However, consider moving all such functionality
          out of the plugin and making the plugin optional.
*/
QUrl QQmlExtensionPlugin::baseUrl() const
{
    Q_D(const QQmlExtensionPlugin);
    return d->baseUrl;
}
#endif

/*!
  \since 6.0

  Override this method to unregister types manually registered in registerTypes.
*/
void QQmlExtensionPlugin::unregisterTypes()
{

}

/*!
    Initializes the extension from the \a uri using the \a engine. Here an application
    plugin might, for example, expose some data or objects to QML,
    as context properties on the engine's root context.
 */
void QQmlExtensionPlugin::initializeEngine(QQmlEngine *engine, const char *uri)
{
    Q_UNUSED(engine);
    Q_UNUSED(uri);
}

/*!
    Initializes the extension from the \a uri using the \a engine. Here an application
    plugin might, for example, expose some data or objects to QML,
    as context properties on the engine's root context.
 */
void QQmlEngineExtensionPlugin::initializeEngine(QQmlEngine *engine, const char *uri)
{
    Q_UNUSED(engine);
    Q_UNUSED(uri);
}

/*!
  \class QQmlExtensionInterface
  \internal
  \inmodule QtQml
*/

/*!
  \class QQmlTypesExtensionInterface
  \internal
  \inmodule QtQml
*/

/*!
  \class QQmlEngineExtensionInterface
  \internal
  \inmodule QtQml
*/


/*!
    \macro Q_IMPORT_QML_PLUGIN(PluginName)
    \since 6.2
    \relates QQmlEngineExtensionPlugin

    Ensures the plugin whose metadata-declaring plugin extension class is named
    \a PluginName is linked into static builds. For the modules created using
    \l qt_add_qml_module, the default plugin extension class name is computed
    from the QML module URI by replacing dots with underscores, unless the
    \c CLASS_NAME argument is specified.

    For example:
    \badcode
        qt_add_qml_module(myplugin
            # The plugin extension class name in this case is my_Company_QmlComponents.
            URI my.Company.QmlComponents
            ...
        )
    \endcode

    \sa Q_IMPORT_PLUGIN
*/

QT_END_NAMESPACE

#include "moc_qqmlextensionplugin.cpp"
