// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.grid.node.local;

import static java.nio.file.Files.readAttributes;
import static java.time.ZoneOffset.UTC;
import static java.time.format.DateTimeFormatter.RFC_1123_DATE_TIME;
import static java.util.Locale.US;
import static java.util.Objects.requireNonNullElseGet;
import static java.util.stream.Collectors.toUnmodifiableSet;
import static org.openqa.selenium.HasDownloads.DownloadedFile;
import static org.openqa.selenium.concurrent.ExecutorServices.shutdownGracefully;
import static org.openqa.selenium.grid.data.Availability.DOWN;
import static org.openqa.selenium.grid.data.Availability.DRAINING;
import static org.openqa.selenium.grid.data.Availability.UP;
import static org.openqa.selenium.grid.node.CapabilityResponseEncoder.getEncoder;
import static org.openqa.selenium.net.Urls.urlDecode;
import static org.openqa.selenium.remote.CapabilityType.ENABLE_DOWNLOADS;
import static org.openqa.selenium.remote.HttpSessionId.getSessionId;
import static org.openqa.selenium.remote.RemoteTags.CAPABILITIES;
import static org.openqa.selenium.remote.RemoteTags.SESSION_ID;
import static org.openqa.selenium.remote.http.Contents.asJson;
import static org.openqa.selenium.remote.http.HttpMethod.DELETE;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalCause;
import com.github.benmanes.caffeine.cache.Ticker;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.net.MediaType;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.NoSuchFileException;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.ImmutableCapabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.NoSuchSessionException;
import org.openqa.selenium.PersistentCapabilities;
import org.openqa.selenium.RetrySessionRequestException;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.concurrent.GuardedRunnable;
import org.openqa.selenium.events.EventBus;
import org.openqa.selenium.grid.data.Availability;
import org.openqa.selenium.grid.data.CreateSessionRequest;
import org.openqa.selenium.grid.data.CreateSessionResponse;
import org.openqa.selenium.grid.data.NodeDrainComplete;
import org.openqa.selenium.grid.data.NodeDrainStarted;
import org.openqa.selenium.grid.data.NodeHeartBeatEvent;
import org.openqa.selenium.grid.data.NodeId;
import org.openqa.selenium.grid.data.NodeStatus;
import org.openqa.selenium.grid.data.Session;
import org.openqa.selenium.grid.data.SessionClosedReason;
import org.openqa.selenium.grid.data.SessionCreatedData;
import org.openqa.selenium.grid.data.SessionCreatedEvent;
import org.openqa.selenium.grid.data.SessionEvent;
import org.openqa.selenium.grid.data.SessionEventData;
import org.openqa.selenium.grid.data.Slot;
import org.openqa.selenium.grid.data.SlotId;
import org.openqa.selenium.grid.jmx.JMXHelper;
import org.openqa.selenium.grid.jmx.ManagedAttribute;
import org.openqa.selenium.grid.jmx.ManagedService;
import org.openqa.selenium.grid.node.ActiveSession;
import org.openqa.selenium.grid.node.HealthCheck;
import org.openqa.selenium.grid.node.Node;
import org.openqa.selenium.grid.node.SessionFactory;
import org.openqa.selenium.grid.node.config.NodeOptions;
import org.openqa.selenium.grid.node.docker.DockerSession;
import org.openqa.selenium.grid.security.Secret;
import org.openqa.selenium.internal.Debug;
import org.openqa.selenium.internal.Either;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.io.FileHandler;
import org.openqa.selenium.io.TemporaryFilesystem;
import org.openqa.selenium.io.Zip;
import org.openqa.selenium.json.Json;
import org.openqa.selenium.remote.Browser;
import org.openqa.selenium.remote.SessionId;
import org.openqa.selenium.remote.http.Contents;
import org.openqa.selenium.remote.http.HttpMethod;
import org.openqa.selenium.remote.http.HttpRequest;
import org.openqa.selenium.remote.http.HttpResponse;
import org.openqa.selenium.remote.tracing.AttributeKey;
import org.openqa.selenium.remote.tracing.AttributeMap;
import org.openqa.selenium.remote.tracing.Span;
import org.openqa.selenium.remote.tracing.Status;
import org.openqa.selenium.remote.tracing.Tracer;

@ManagedService(
    objectName = "org.seleniumhq.grid:type=Node,name=LocalNode",
    description = "Node running the webdriver sessions.")
public class LocalNode extends Node implements Closeable {

  private static final Json JSON = new Json();
  private static final Logger LOG = Logger.getLogger(LocalNode.class.getName());
  private static final DateTimeFormatter HTTP_DATE_FORMAT = RFC_1123_DATE_TIME.withLocale(US);

  private final EventBus bus;
  private final URI externalUri;
  private final URI gridUri;
  private final Duration heartbeatPeriod;
  private final HealthCheck healthCheck;
  private final int maxSessionCount;
  private final int configuredSessionCount;
  private final boolean cdpEnabled;
  private final boolean managedDownloadsEnabled;
  private final int connectionLimitPerSession;

  private final boolean bidiEnabled;
  private final boolean drainAfterSessions;
  private final List<SessionSlot> factories;
  private final Cache<SessionId, SessionSlot> currentSessions;
  private final Cache<SessionId, TemporaryFilesystem> uploadsTempFileSystem;
  private final Cache<SessionId, TemporaryFilesystem> downloadsTempFileSystem;
  private final AtomicInteger pendingSessions = new AtomicInteger();
  private final AtomicInteger sessionCount = new AtomicInteger();
  // Tracks sessions that are reserved (pending creation) or active, used for maxSessionCount check
  private final AtomicInteger reservedOrActiveSessionCount = new AtomicInteger();
  // Tracks consecutive session creation failures to mark node DOWN if threshold exceeded
  private final AtomicInteger consecutiveSessionFailures = new AtomicInteger();
  private final int nodeDownFailureThreshold;
  private final Runnable shutdown;
  private final ReadWriteLock drainLock = new ReentrantReadWriteLock();

  protected LocalNode(
      Tracer tracer,
      EventBus bus,
      URI uri,
      URI gridUri,
      HealthCheck healthCheck,
      int maxSessionCount,
      int drainAfterSessionCount,
      boolean cdpEnabled,
      boolean bidiEnabled,
      Ticker ticker,
      Duration sessionTimeout,
      Duration heartbeatPeriod,
      List<SessionSlot> factories,
      Secret registrationSecret,
      boolean managedDownloadsEnabled,
      int connectionLimitPerSession,
      int nodeDownFailureThreshold) {
    super(
        tracer,
        new NodeId(UUID.randomUUID()),
        uri,
        registrationSecret,
        Require.positive(sessionTimeout));

    this.bus = Require.nonNull("Event bus", bus);

    this.externalUri = Require.nonNull("Remote node URI", uri);
    this.gridUri = Require.nonNull("Grid URI", gridUri);
    this.maxSessionCount =
        Math.min(Require.positive("Max session count", maxSessionCount), factories.size());
    this.heartbeatPeriod = heartbeatPeriod;
    this.factories = List.copyOf(factories);
    Require.nonNull("Registration secret", registrationSecret);
    this.configuredSessionCount = drainAfterSessionCount;
    this.drainAfterSessions = this.configuredSessionCount > 0;
    this.sessionCount.set(drainAfterSessionCount);
    this.cdpEnabled = cdpEnabled;
    this.bidiEnabled = bidiEnabled;
    this.managedDownloadsEnabled = managedDownloadsEnabled;
    this.connectionLimitPerSession = connectionLimitPerSession;
    // Use 0 to disable the failure threshold feature (unlimited retries)
    this.nodeDownFailureThreshold = nodeDownFailureThreshold;

    this.healthCheck =
        healthCheck == null
            ? () -> {
              NodeStatus status = getStatus();
              return new HealthCheck.Result(
                  status.getAvailability(),
                  String.format("%s is %s", uri, status.getAvailability()));
            }
            : healthCheck;

    // Do not clear this cache automatically using a timer.
    // It will be explicitly cleaned up, as and when "currentSessions" is auto cleaned.
    this.uploadsTempFileSystem =
        Caffeine.newBuilder()
            .removalListener(
                (SessionId key, TemporaryFilesystem tempFS, RemovalCause cause) -> {
                  Optional.ofNullable(tempFS)
                      .ifPresent(
                          fs -> {
                            fs.deleteTemporaryFiles();
                            fs.deleteBaseDir();
                          });
                })
            .build();

    // Do not clear this cache automatically using a timer.
    // It will be explicitly cleaned up, as and when "currentSessions" is auto cleaned.
    this.downloadsTempFileSystem =
        Caffeine.newBuilder()
            .removalListener(
                (SessionId key, TemporaryFilesystem tempFS, RemovalCause cause) -> {
                  Optional.ofNullable(tempFS)
                      .ifPresent(
                          fs -> {
                            fs.deleteTemporaryFiles();
                            fs.deleteBaseDir();
                          });
                })
            .build();

    this.currentSessions =
        Caffeine.newBuilder()
            .expireAfterAccess(sessionTimeout)
            .ticker(ticker)
            .removalListener(this::stopTimedOutSession)
            .build();

    ScheduledExecutorService sessionCleanupNodeService =
        Executors.newSingleThreadScheduledExecutor(
            r -> {
              Thread thread = new Thread(r);
              thread.setDaemon(true);
              thread.setName("Local Node - Session Cleanup " + externalUri);
              return thread;
            });
    sessionCleanupNodeService.scheduleAtFixedRate(
        GuardedRunnable.guard(currentSessions::cleanUp), 30, 30, TimeUnit.SECONDS);

    ScheduledExecutorService uploadTempFileCleanupNodeService =
        Executors.newSingleThreadScheduledExecutor(
            r -> {
              Thread thread = new Thread(r);
              thread.setDaemon(true);
              thread.setName("UploadTempFile Cleanup Node " + externalUri);
              return thread;
            });
    uploadTempFileCleanupNodeService.scheduleAtFixedRate(
        GuardedRunnable.guard(uploadsTempFileSystem::cleanUp), 30, 30, TimeUnit.SECONDS);

    ScheduledExecutorService downloadTempFileCleanupNodeService =
        Executors.newSingleThreadScheduledExecutor(
            r -> {
              Thread thread = new Thread(r);
              thread.setDaemon(true);
              thread.setName("DownloadTempFile Cleanup Node " + externalUri);
              return thread;
            });
    downloadTempFileCleanupNodeService.scheduleAtFixedRate(
        GuardedRunnable.guard(downloadsTempFileSystem::cleanUp), 30, 30, TimeUnit.SECONDS);

    ScheduledExecutorService heartbeatNodeService =
        Executors.newSingleThreadScheduledExecutor(
            r -> {
              Thread thread = new Thread(r);
              thread.setDaemon(true);
              thread.setName("HeartBeat Node " + externalUri);
              return thread;
            });
    heartbeatNodeService.scheduleAtFixedRate(
        GuardedRunnable.guard(() -> bus.fire(new NodeHeartBeatEvent(getStatus()))),
        heartbeatPeriod.getSeconds(),
        heartbeatPeriod.getSeconds(),
        TimeUnit.SECONDS);

    shutdown =
        () -> {
          if (heartbeatNodeService.isShutdown()) return;

          shutdownGracefully(
              "Local Node - Session Cleanup " + externalUri, sessionCleanupNodeService);
          shutdownGracefully(
              "UploadTempFile Cleanup Node " + externalUri, uploadTempFileCleanupNodeService);
          shutdownGracefully(
              "DownloadTempFile Cleanup Node " + externalUri, downloadTempFileCleanupNodeService);
          shutdownGracefully("HeartBeat Node " + externalUri, heartbeatNodeService);

          // ensure we do not leak running browsers
          currentSessions.invalidateAll();
          currentSessions.cleanUp();
        };

    Runtime.getRuntime()
        .addShutdownHook(
            new Thread(
                () -> {
                  stopAllSessions();
                  drain();
                }));

    new JMXHelper().register(this);
  }

  @Override
  public void close() {
    shutdown.run();
  }

  private void stopTimedOutSession(SessionId id, SessionSlot slot, RemovalCause cause) {
    try (Span span = tracer.getCurrentContext().createSpan("node.stop_session")) {
      AttributeMap attributeMap = tracer.createAttributeMap();
      attributeMap.put(AttributeKey.LOGGER_CLASS.getKey(), getClass().getName());
      if (id != null && slot != null) {
        attributeMap.put("node.id", getId().toString());
        attributeMap.put("session.slotId", slot.getId().toString());
        attributeMap.put("session.id", id.toString());
        attributeMap.put("session.timeout_in_seconds", getSessionTimeout().toSeconds());
        attributeMap.put("session.remove.cause", cause.name());

        // Determine the SessionClosedReason based on RemovalCause
        SessionClosedReason closeReason;
        if (cause == RemovalCause.EXPIRED) {
          closeReason = SessionClosedReason.TIMEOUT;
          // Session is timing out, stopping it by sending a DELETE
          LOG.log(Level.INFO, () -> String.format("Session id %s timed out, stopping...", id));
          span.setStatus(Status.CANCELLED);
          span.addEvent(String.format("Stopping the the timed session %s", id), attributeMap);
        } else {
          closeReason = SessionClosedReason.QUIT_COMMAND;
          LOG.log(Level.INFO, () -> String.format("Session id %s is stopping on demand...", id));
          span.addEvent(String.format("Stopping the session %s on demand", id), attributeMap);
        }
        if (cause == RemovalCause.EXPIRED) {
          try {
            slot.execute(new HttpRequest(DELETE, "/session/" + id));
          } catch (Exception e) {
            LOG.log(
                Level.WARNING, String.format("Exception while trying to stop session %s", id), e);
            span.setStatus(Status.INTERNAL);
            span.addEvent(
                String.format("Exception while trying to stop session %s", id), attributeMap);
          }
        }
        // Attempt to stop the session with the appropriate reason and node context
        slot.stop(closeReason, getId(), externalUri);
        // Decrement the reserved/active session counter
        reservedOrActiveSessionCount.decrementAndGet();
        // Decrement pending sessions if Node is draining
        if (this.isDraining()) {
          int done = pendingSessions.decrementAndGet();
          attributeMap.put("current.session.count", done);
          attributeMap.put("node.drain_after_session_count", this.configuredSessionCount);
          if (done <= 0) {
            LOG.info("Node draining complete!");
            bus.fire(new NodeDrainComplete(this.getId()));
            span.addEvent("Node draining complete!", attributeMap);
          }
        }
      } else {
        LOG.log(Debug.getDebugLogLevel(), "Received stop session notification with null values");
        span.setStatus(Status.INVALID_ARGUMENT);
        span.addEvent("Received stop session notification with null values", attributeMap);
      }
    }
  }

  public static Builder builder(
      Tracer tracer, EventBus bus, URI uri, URI gridUri, Secret registrationSecret) {
    return new Builder(tracer, bus, uri, gridUri, registrationSecret);
  }

  @Override
  public boolean isReady() {
    return bus.isReady();
  }

  @VisibleForTesting
  @ManagedAttribute(name = "CurrentSessions")
  public int getCurrentSessionCount() {
    // we need the exact size, see javadoc of Cache.size
    long n = currentSessions.asMap().values().stream().count();
    // It seems wildly unlikely we'll overflow an int
    return Math.toIntExact(n);
  }

  @ManagedAttribute(name = "MaxSessions")
  public int getMaxSessionCount() {
    return maxSessionCount;
  }

  @ManagedAttribute(name = "Status")
  public Availability getAvailability() {
    if (nodeDownFailureThreshold > 0
        && consecutiveSessionFailures.get() >= nodeDownFailureThreshold) {
      return DOWN;
    }
    return isDraining() ? DRAINING : UP;
  }

  @ManagedAttribute(name = "TotalSlots")
  public int getTotalSlots() {
    return factories.size();
  }

  @ManagedAttribute(name = "UsedSlots")
  public long getUsedSlots() {
    return factories.stream().filter(sessionSlot -> !sessionSlot.isAvailable()).count();
  }

  @ManagedAttribute(name = "Load")
  public float getLoad() {
    long inUse = factories.stream().filter(sessionSlot -> !sessionSlot.isAvailable()).count();
    return inUse / (float) maxSessionCount * 100f;
  }

  @ManagedAttribute(name = "ConsecutiveSessionFailures")
  public int getConsecutiveSessionFailures() {
    return consecutiveSessionFailures.get();
  }

  /**
   * Resets the consecutive session creation failure counter. This can be used to recover a node
   * that was marked as DOWN due to exceeding the failure threshold, for example after external
   * intervention has resolved the underlying issue.
   */
  public void resetConsecutiveSessionFailures() {
    int previousValue = consecutiveSessionFailures.getAndSet(0);
    if (previousValue > 0) {
      LOG.info(
          String.format(
              "Consecutive session failure counter reset from %d to 0. Node availability restored.",
              previousValue));
    }
  }

  @ManagedAttribute(name = "RemoteNodeUri")
  public URI getExternalUri() {
    return this.getUri();
  }

  @ManagedAttribute(name = "GridUri")
  public URI getGridUri() {
    return this.gridUri;
  }

  @ManagedAttribute(name = "NodeId")
  public String getNodeId() {
    return getId().toString();
  }

  @Override
  public boolean isSupporting(Capabilities capabilities) {
    return factories.parallelStream().anyMatch(factory -> factory.test(capabilities));
  }

  @Override
  public Either<WebDriverException, CreateSessionResponse> newSession(
      CreateSessionRequest sessionRequest) {
    Require.nonNull("Session request", sessionRequest);

    // Early check before acquiring lock (fast path for draining nodes)
    if (isDraining()) {
      return Either.left(
          new RetrySessionRequestException("The node is draining. Cannot accept new sessions."));
    }

    // Early check for failure threshold (fast path for unhealthy nodes)
    if (nodeDownFailureThreshold > 0
        && consecutiveSessionFailures.get() >= nodeDownFailureThreshold) {
      return Either.left(
          new RetrySessionRequestException(
              "The node is marked as DOWN due to exceeding the failure threshold. Cannot accept new"
                  + " sessions."));
    }

    Lock lock = drainLock.readLock();
    lock.lock();

    try (Span span = tracer.getCurrentContext().createSpan("node.new_session")) {
      AttributeMap attributeMap = tracer.createAttributeMap();
      attributeMap.put(AttributeKey.LOGGER_CLASS.getKey(), getClass().getName());
      attributeMap.put(
          "session.request.capabilities", sessionRequest.getDesiredCapabilities().toString());
      attributeMap.put(
          "session.request.downstreamdialect", sessionRequest.getDownstreamDialects().toString());

      // Re-check after acquiring lock (double-checked locking pattern)
      if (isDraining()) {
        span.setStatus(
            Status.UNAVAILABLE.withDescription(
                "The node is draining. Cannot accept new sessions."));
        return Either.left(
            new RetrySessionRequestException("The node is draining. Cannot accept new sessions."));
      }

      // Re-check failure threshold after acquiring lock
      if (nodeDownFailureThreshold > 0
          && consecutiveSessionFailures.get() >= nodeDownFailureThreshold) {
        span.setStatus(
            Status.UNAVAILABLE.withDescription(
                "The node is marked as DOWN due to exceeding the failure threshold."));
        return Either.left(
            new RetrySessionRequestException(
                "The node is marked as DOWN due to exceeding the failure threshold. "
                    + "Cannot accept new sessions."));
      }

      // Atomically check and reserve a session slot to prevent exceeding maxSessionCount
      // This fixes the race condition where multiple threads could pass the count check
      int currentCount = reservedOrActiveSessionCount.incrementAndGet();
      span.setAttribute("current.session.count", currentCount);
      attributeMap.put("current.session.count", currentCount);

      if (currentCount > maxSessionCount) {
        reservedOrActiveSessionCount.decrementAndGet();
        span.setAttribute(AttributeKey.ERROR.getKey(), true);
        span.setStatus(Status.RESOURCE_EXHAUSTED);
        attributeMap.put("max.session.count", maxSessionCount);
        span.addEvent("Max session count reached", attributeMap);
        return Either.left(new RetrySessionRequestException("Max session count reached."));
      }

      // Identify possible slots to use as quickly as possible to enable concurrent session starting
      // Uses lock-free tryReserve() for better concurrency - no global lock needed
      SessionSlot slotToUse = null;
      Capabilities desiredCaps = sessionRequest.getDesiredCapabilities();
      for (SessionSlot factory : factories) {
        if (factory.tryReserve(desiredCaps)) {
          slotToUse = factory;
          break;
        }
      }

      if (slotToUse == null) {
        reservedOrActiveSessionCount.decrementAndGet();
        span.setAttribute(AttributeKey.ERROR.getKey(), true);
        span.setStatus(Status.NOT_FOUND);
        span.addEvent("No slot matched the requested capabilities. ", attributeMap);
        return Either.left(
            new RetrySessionRequestException("No slot matched the requested capabilities."));
      }

      if (!decrementSessionCount()) {
        slotToUse.release();
        reservedOrActiveSessionCount.decrementAndGet();
        span.setAttribute(AttributeKey.ERROR.getKey(), true);
        span.setStatus(Status.RESOURCE_EXHAUSTED);
        attributeMap.put("drain.after.session.count", configuredSessionCount);
        span.addEvent("Drain after session count reached", attributeMap);
        return Either.left(new RetrySessionRequestException("Drain after session count reached."));
      }

      Capabilities desiredCapabilities = sessionRequest.getDesiredCapabilities();
      TemporaryFilesystem downloadsTfs;
      if (managedDownloadsRequested(desiredCapabilities)) {
        UUID uuidForSessionDownloads = UUID.randomUUID();

        downloadsTfs =
            TemporaryFilesystem.getTmpFsBasedOn(
                TemporaryFilesystem.getDefaultTmpFS()
                    .createTempDir("uuid", uuidForSessionDownloads.toString()));

        Capabilities enhanced = setDownloadsDirectory(downloadsTfs, desiredCapabilities);
        enhanced = desiredCapabilities.merge(enhanced);
        sessionRequest =
            new CreateSessionRequest(
                sessionRequest.getDownstreamDialects(), enhanced, sessionRequest.getMetadata());
      } else {
        downloadsTfs = null;
      }

      Either<WebDriverException, ActiveSession> possibleSession = slotToUse.apply(sessionRequest);

      if (possibleSession.isRight()) {
        ActiveSession session = possibleSession.right();
        if (downloadsTfs != null) {
          downloadsTempFileSystem.put(session.getId(), downloadsTfs);
        }
        currentSessions.put(session.getId(), slotToUse);

        // Reset consecutive failure counter on successful session creation
        consecutiveSessionFailures.set(0);

        SessionId sessionId = session.getId();
        Capabilities caps = session.getCapabilities();
        SESSION_ID.accept(span, sessionId);
        CAPABILITIES.accept(span, caps);
        String downstream = session.getDownstreamDialect().toString();
        String upstream = session.getUpstreamDialect().toString();
        String sessionUri = session.getUri().toString();
        span.setAttribute(AttributeKey.DOWNSTREAM_DIALECT.getKey(), downstream);
        span.setAttribute(AttributeKey.UPSTREAM_DIALECT.getKey(), upstream);
        span.setAttribute(AttributeKey.SESSION_URI.getKey(), sessionUri);

        // The session we return has to look like it came from the node, since we might be dealing
        // with a webdriver implementation that only accepts connections from localhost
        Session externalSession =
            createExternalSession(
                session,
                externalUri,
                slotToUse.isSupportingCdp(),
                slotToUse.isSupportingBiDi(),
                desiredCapabilities);

        String sessionCreatedMessage = "Session created by the Node";
        LOG.info(
            String.format(
                "%s. Id: %s, Caps: %s",
                sessionCreatedMessage, sessionId, externalSession.getCapabilities()));

        // Create session data for events and listeners
        SessionCreatedData createdData =
            new SessionCreatedData(
                sessionId,
                getId(),
                externalUri,
                session.getUri(),
                externalSession.getCapabilities(),
                slotToUse.getStereotype(),
                externalSession.getStartTime());

        // Fire session created event for sidecar services
        bus.fire(new SessionCreatedEvent(createdData));

        return Either.right(
            new CreateSessionResponse(
                externalSession,
                getEncoder(session.getDownstreamDialect()).apply(externalSession)));
      } else {
        slotToUse.release();
        reservedOrActiveSessionCount.decrementAndGet();
        // Restore session count that was decremented earlier, so node doesn't drain prematurely
        restoreSessionCount();

        // Track consecutive session creation failures
        int failures = consecutiveSessionFailures.incrementAndGet();
        if (nodeDownFailureThreshold > 0 && failures >= nodeDownFailureThreshold) {
          LOG.warning(
              String.format(
                  "Node has reached the failure threshold (%d consecutive failures). "
                      + "Node will be marked as DOWN.",
                  nodeDownFailureThreshold));
        }

        if (downloadsTfs != null) {
          downloadsTfs.deleteTemporaryFiles();
          downloadsTfs.deleteBaseDir();
        }
        span.setAttribute(AttributeKey.ERROR.getKey(), true);
        span.setStatus(Status.ABORTED);
        span.addEvent("Unable to create session with the driver", attributeMap);
        return Either.left(possibleSession.left());
      }
    } finally {
      lock.unlock();
      checkSessionCount();
    }
  }

  private boolean managedDownloadsRequested(Capabilities capabilities) {
    Object downloadsEnabled = capabilities.getCapability(ENABLE_DOWNLOADS);
    return managedDownloadsEnabled
        && downloadsEnabled != null
        && Boolean.parseBoolean(downloadsEnabled.toString());
  }

  private Capabilities setDownloadsDirectory(TemporaryFilesystem downloadsTfs, Capabilities caps) {
    File tempDir = downloadsTfs.createTempDir("download", "");
    if (Browser.CHROME.is(caps) || Browser.EDGE.is(caps)) {
      Map<String, Serializable> map =
          Map.of(
              "download.prompt_for_download",
              false,
              "download.default_directory",
              tempDir.getAbsolutePath(),
              "savefile.default_directory",
              tempDir.getAbsolutePath());
      String optionsKey = Browser.CHROME.is(caps) ? "goog:chromeOptions" : "ms:edgeOptions";
      return appendPrefs(caps, optionsKey, map);
    }
    if (Browser.FIREFOX.is(caps)) {
      Map<String, Serializable> map =
          Map.of(
              "browser.download.folderList", 2, "browser.download.dir", tempDir.getAbsolutePath());
      return appendPrefs(caps, "moz:firefoxOptions", map);
    }
    return caps;
  }

  @SuppressWarnings("unchecked")
  private Capabilities appendPrefs(
      Capabilities caps, String optionsKey, Map<String, Serializable> map) {
    if (caps.getCapability(optionsKey) == null) {
      MutableCapabilities mutableCaps = new MutableCapabilities();
      mutableCaps.setCapability(optionsKey, new HashMap<>());
      caps = caps.merge(mutableCaps);
    }
    Map<String, Object> currentOptions = (Map<String, Object>) caps.getCapability(optionsKey);

    ((Map<String, Serializable>) currentOptions.computeIfAbsent("prefs", k -> new HashMap<>()))
        .putAll(map);
    return caps;
  }

  @Override
  public boolean isSessionOwner(SessionId id) {
    Require.nonNull("Session ID", id);
    return currentSessions.getIfPresent(id) != null;
  }

  @Override
  public boolean tryAcquireConnection(SessionId id) throws NoSuchSessionException {
    SessionSlot slot = currentSessions.getIfPresent(id);

    if (slot == null) {
      return false;
    }

    if (connectionLimitPerSession == -1) {
      // no limit
      return true;
    }

    AtomicLong counter = slot.getConnectionCounter();

    if (connectionLimitPerSession > counter.getAndIncrement()) {
      return true;
    }

    // ensure a rejected connection will not be counted
    counter.getAndDecrement();
    return false;
  }

  @Override
  public void releaseConnection(SessionId id) {
    SessionSlot slot = currentSessions.getIfPresent(id);

    if (slot == null) {
      return;
    }

    if (connectionLimitPerSession == -1) {
      // no limit
      return;
    }

    AtomicLong counter = slot.getConnectionCounter();

    counter.decrementAndGet();
  }

  @Override
  public Session getSession(SessionId id) throws NoSuchSessionException {
    Require.nonNull("Session ID", id);

    SessionSlot slot = currentSessions.getIfPresent(id);
    if (slot == null) {
      throw new NoSuchSessionException("Cannot find session with id: " + id);
    }

    return createExternalSession(
        slot.getSession(),
        externalUri,
        slot.isSupportingCdp(),
        slot.isSupportingBiDi(),
        slot.getSession().getCapabilities());
  }

  @Override
  public TemporaryFilesystem getUploadsFilesystem(SessionId id) throws IOException {
    return uploadsTempFileSystem.get(
        id,
        key ->
            TemporaryFilesystem.getTmpFsBasedOn(
                TemporaryFilesystem.getDefaultTmpFS().createTempDir("session", id.toString())));
  }

  @Override
  public TemporaryFilesystem getDownloadsFilesystem(SessionId sessionId) {
    return downloadsTempFileSystem.getIfPresent(sessionId);
  }

  @Override
  public HttpResponse executeWebDriverCommand(HttpRequest req) {
    // True enough to be good enough
    SessionId id =
        getSessionId(req.getUri())
            .map(SessionId::new)
            .orElseThrow(() -> new NoSuchSessionException("Cannot find session: " + req));

    SessionSlot slot = currentSessions.getIfPresent(id);
    if (slot == null) {
      throw new NoSuchSessionException("Cannot find session with id: " + id);
    }

    HttpResponse toReturn = slot.execute(req);
    if (req.getMethod() == DELETE && req.getUri().equals("/session/" + id)) {
      stop(id);
    }
    return toReturn;
  }

  @Override
  public HttpResponse downloadFile(HttpRequest req, SessionId id) {
    // When the session is running in a Docker container, the download file command
    // needs to be forwarded to the container as well.
    SessionSlot slot = currentSessions.getIfPresent(id);
    if (slot != null && slot.getSession() instanceof DockerSession) {
      return executeWebDriverCommand(req);
    }
    if (!this.managedDownloadsEnabled) {
      String msg =
          "Please enable management of downloads via the command line arg "
              + "[--enable-managed-downloads] and restart the node";
      throw new WebDriverException(msg);
    }
    TemporaryFilesystem tempFS = downloadsTempFileSystem.getIfPresent(id);
    if (tempFS == null) {
      String msg =
          "Cannot find downloads file system for session id: "
              + id
              + " — ensure downloads are enabled in the options class when requesting a session.";
      throw new WebDriverException(msg);
    }
    File downloadsDirectory =
        Optional.ofNullable(tempFS.getBaseDir().listFiles()).orElse(new File[] {})[0];

    try {
      if (req.getMethod().equals(HttpMethod.GET) && req.getUri().endsWith("/se/files")) {
        return listDownloadedFiles(downloadsDirectory);
      }
      if (req.getMethod().equals(HttpMethod.GET)) {
        // Left here for backward compatibility.
        // Remove this IF in Selenium 4.41, 4.42 or 4.43
        return getDownloadedFile(downloadsDirectory, extractFileName(req));
      }
      if (req.getMethod().equals(HttpMethod.DELETE)) {
        return deleteDownloadedFile(downloadsDirectory);
      }
      return getDownloadedFile(req, downloadsDirectory);
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }

  private String extractFileName(HttpRequest req) {
    return extractFileName(req.getUri());
  }

  String extractFileName(String uri) {
    String prefix = "/se/files/";
    int index = uri.lastIndexOf(prefix);
    if (index < 0) {
      throw new IllegalArgumentException("Unexpected URL for downloading a file: " + uri);
    }
    return urlDecode(uri.substring(index + prefix.length())).replace(' ', '+');
  }

  /** User wants to list files that can be downloaded */
  private HttpResponse listDownloadedFiles(File downloadsDirectory) {
    File[] files = Optional.ofNullable(downloadsDirectory.listFiles()).orElse(new File[] {});
    List<String> fileNames = Arrays.stream(files).map(File::getName).collect(Collectors.toList());
    List<DownloadedFile> fileInfos =
        Arrays.stream(files)
            .map(this::getFileInfo)
            .filter(file -> file.getLastModifiedTime() > 0)
            .collect(Collectors.toList());

    Map<String, Object> data =
        Map.of(
            "names", fileNames,
            "files", fileInfos);
    Map<String, Map<String, Object>> result = Map.of("value", data);
    return new HttpResponse().setContent(asJson(result));
  }

  private DownloadedFile getFileInfo(File file) {
    try {
      BasicFileAttributes attributes = readAttributes(file.toPath(), BasicFileAttributes.class);
      return new DownloadedFile(
          file.getName(),
          attributes.creationTime().toMillis(),
          attributes.lastModifiedTime().toMillis(),
          attributes.size());
    } catch (NoSuchFileException e) {
      return new DownloadedFile(file.getName(), -1, -1, -1);
    } catch (IOException e) {
      throw new UncheckedIOException("Failed to get file attributes: " + file.getAbsolutePath(), e);
    }
  }

  private HttpResponse getDownloadedFile(HttpRequest req, File downloadsDirectory)
      throws IOException {
    String raw = req.contentAsString();
    if (raw.isEmpty()) {
      throw new WebDriverException(
          "Please specify file to download in payload as {\"name\": \"fileToDownload\"}");
    }
    Map<String, Object> incoming = JSON.toType(raw, Json.MAP_TYPE);
    String filename =
        Optional.ofNullable(incoming.get("name"))
            .map(Object::toString)
            .orElseThrow(
                () ->
                    new WebDriverException(
                        "Please specify file to download in payload as {\"name\":"
                            + " \"fileToDownload\"}"));
    File file = findDownloadedFile(downloadsDirectory, filename);
    String contentType =
        requireNonNullElseGet(
            (String) incoming.get("format"), () -> MediaType.JSON_UTF_8.toString());

    if (MediaType.OCTET_STREAM.toString().equalsIgnoreCase(contentType)) {
      return fileAsBinaryResponse(file);
    }

    String content = Zip.zip(file);
    Map<String, Object> data =
        Map.of(
            "filename", filename,
            "file", getFileInfo(file),
            "contents", content);
    Map<String, Map<String, Object>> result = Map.of("value", data);
    return new HttpResponse().setContent(asJson(result));
  }

  /** Left here for backward compatibility. Remove this method in Selenium 4.41, 4.42 or 4.43 */
  @Deprecated
  private HttpResponse getDownloadedFile(File downloadsDirectory, String fileName)
      throws IOException {
    if (fileName.isEmpty()) {
      throw new WebDriverException("Please specify file to download in URL");
    }
    File file = findDownloadedFile(downloadsDirectory, fileName);
    return fileAsBinaryResponse(file);
  }

  private HttpResponse fileAsBinaryResponse(File file) throws IOException {
    BasicFileAttributes attributes = readAttributes(file.toPath(), BasicFileAttributes.class);
    return new HttpResponse()
        .setHeader("Content-Type", MediaType.OCTET_STREAM.toString())
        .setHeader("Content-Length", String.valueOf(attributes.size()))
        .setHeader("Last-Modified", lastModifiedHeader(attributes.lastModifiedTime()))
        .setContent(Contents.file(file));
  }

  private String lastModifiedHeader(FileTime fileTime) {
    return HTTP_DATE_FORMAT.format(fileTime.toInstant().atZone(UTC));
  }

  private File findDownloadedFile(File downloadsDirectory, String filename)
      throws WebDriverException {
    List<File> matchingFiles =
        List.of(
            requireNonNullElseGet(
                downloadsDirectory.listFiles((dir, name) -> name.equals(filename)),
                () -> new File[0]));
    if (matchingFiles.isEmpty()) {
      List<File> files = downloadedFiles(downloadsDirectory);
      throw new WebDriverException(
          String.format(
              "Cannot find file [%s] in directory %s. Found %s files: %s.",
              filename, downloadsDirectory.getAbsolutePath(), files.size(), files));
    }
    if (matchingFiles.size() != 1) {
      throw new WebDriverException(
          String.format(
              "Expected there to be only 1 file. Found %s files: %s.",
              matchingFiles.size(), matchingFiles));
    }
    return matchingFiles.get(0);
  }

  private static List<File> downloadedFiles(File downloadsDirectory) {
    File[] files = requireNonNullElseGet(downloadsDirectory.listFiles(), () -> new File[0]);
    return List.of(files);
  }

  private HttpResponse deleteDownloadedFile(File downloadsDirectory) {
    File[] files = Optional.ofNullable(downloadsDirectory.listFiles()).orElse(new File[] {});
    for (File file : files) {
      FileHandler.delete(file);
    }
    Map<String, Object> toReturn = new HashMap<>();
    toReturn.put("value", null);
    return new HttpResponse().setContent(asJson(toReturn));
  }

  @Override
  public HttpResponse uploadFile(HttpRequest req, SessionId id) {

    // When the session is running in a Docker container, the upload file command
    // needs to be forwarded to the container as well.
    SessionSlot slot = currentSessions.getIfPresent(id);
    if (slot != null && slot.getSession() instanceof DockerSession) {
      return executeWebDriverCommand(req);
    }

    Map<String, Object> incoming = JSON.toType(req.contentAsString(), Json.MAP_TYPE);

    File tempDir;
    try {
      TemporaryFilesystem tempFS = getUploadsFilesystem(id);
      tempDir = tempFS.createTempDir("upload", "file");

      Zip.unzip((String) incoming.get("file"), tempDir);
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
    // Select the first file
    File[] allFiles = tempDir.listFiles();
    if (allFiles == null) {
      throw new WebDriverException(
          String.format("Cannot access temporary directory for uploaded files %s", tempDir));
    }
    if (allFiles.length != 1) {
      throw new WebDriverException(
          String.format("Expected there to be only 1 file. There were: %s", allFiles.length));
    }

    Map<String, Object> result = Map.of("value", allFiles[0].getAbsolutePath());

    return new HttpResponse().setContent(asJson(result));
  }

  @Override
  public HttpResponse fireSessionEvent(HttpRequest req, SessionId id) {
    Require.nonNull("Session ID", id);

    // Verify session exists
    SessionSlot slot = currentSessions.getIfPresent(id);
    if (slot == null) {
      throw new NoSuchSessionException("Cannot find session with id: " + id);
    }

    // Parse the event data from request
    Map<String, Object> incoming = JSON.toType(req.contentAsString(), Json.MAP_TYPE);
    String eventType = (String) incoming.get("eventType");
    if (eventType == null || eventType.isEmpty()) {
      throw new WebDriverException(
          "Event type is required. Please provide 'eventType' in payload.");
    }

    Object rawPayload = incoming.get("payload");
    Map<String, Object> payload =
        (rawPayload instanceof Map) ? (Map<String, Object>) rawPayload : Map.of();

    // Create event data with node context
    SessionEventData eventData =
        SessionEventData.create(id, eventType, payload).withNodeContext(getId(), externalUri);

    // Fire event via EventBus for sidecar services
    bus.fire(new SessionEvent(eventData));

    LOG.log(
        Level.FINE,
        () -> String.format("Session event fired: type=%s, sessionId=%s", eventType, id));

    // Return success response
    Map<String, Object> responseData =
        Map.of(
            "success",
            true,
            "eventType",
            eventType,
            "timestamp",
            eventData.getTimestamp().toString());
    Map<String, Object> result = Map.of("value", responseData);
    return new HttpResponse().setContent(asJson(result));
  }

  @Override
  public void stop(SessionId id) throws NoSuchSessionException {
    Require.nonNull("Session ID", id);

    if (downloadsTempFileSystem.getIfPresent(id) != null) {
      downloadsTempFileSystem.invalidate(id);
    }
    if (uploadsTempFileSystem.getIfPresent(id) != null) {
      uploadsTempFileSystem.invalidate(id);
    }

    SessionSlot slot = currentSessions.getIfPresent(id);
    if (slot == null) {
      throw new NoSuchSessionException("Cannot find session with id: " + id);
    }

    currentSessions.invalidate(id);
  }

  private void stopAllSessions() {
    LOG.info("Trying to stop all running sessions before shutting down...");
    currentSessions.invalidateAll();
  }

  private Session createExternalSession(
      ActiveSession other,
      URI externalUri,
      boolean isSupportingCdp,
      boolean isSupportingBiDi,
      Capabilities requestCapabilities) {
    // We merge the session request capabilities and the session ones to keep the values sent
    // by the user in the session information
    Capabilities toUse =
        ImmutableCapabilities.copyOf(requestCapabilities.merge(other.getCapabilities()));

    // Add se:cdp if necessary to send the cdp url back
    if ((isSupportingCdp || toUse.getCapability("se:cdp") != null) && cdpEnabled) {
      String cdpPath = String.format("/session/%s/se/cdp", other.getId());
      toUse = new PersistentCapabilities(toUse).setCapability("se:cdp", rewrite(cdpPath));
    } else {
      // Remove any se:cdp* from the response, CDP is not supported nor enabled
      MutableCapabilities cdpFiltered = new MutableCapabilities();
      toUse
          .asMap()
          .forEach(
              (key, value) -> {
                if (!key.startsWith("se:cdp")) {
                  cdpFiltered.setCapability(key, value);
                }
              });
      toUse = new PersistentCapabilities(cdpFiltered).setCapability("se:cdpEnabled", false);
    }

    // Check if the user wants to use BiDi
    // This will be null if the user has not set the capability.
    Object webSocketUrl = toUse.getCapability("webSocketUrl");

    // In case of Firefox versions that do not support webSocketUrl, it returns the capability as it
    // is i.e. boolean value. So need to check if it is a string.
    // Check if the Node supports BiDi and if the client wants to use BiDi.
    boolean bidiSupported = isSupportingBiDi && (webSocketUrl instanceof String);
    if (bidiSupported && bidiEnabled) {
      String biDiUrl = (String) other.getCapabilities().getCapability("webSocketUrl");
      URI uri;
      try {
        uri = new URI(biDiUrl);
      } catch (URISyntaxException e) {
        throw new IllegalArgumentException("Unable to create URI from " + biDiUrl);
      }
      String bidiPath = String.format("/session/%s/se/bidi", other.getId());
      toUse =
          new PersistentCapabilities(toUse)
              .setCapability("se:gridWebSocketUrl", uri)
              .setCapability("webSocketUrl", rewrite(bidiPath));
    } else {
      // Remove any "webSocketUrl" from the response, BiDi is not supported nor enabled
      MutableCapabilities bidiFiltered = new MutableCapabilities();
      toUse
          .asMap()
          .forEach(
              (key, value) -> {
                if (!key.startsWith("webSocketUrl")) {
                  bidiFiltered.setCapability(key, value);
                }
              });
      toUse = new PersistentCapabilities(bidiFiltered).setCapability("se:bidiEnabled", false);
    }

    // If enabled, set the VNC endpoint for live view
    boolean isVncEnabled = toUse.getCapability("se:vncLocalAddress") != null;
    if (isVncEnabled) {
      String vncPath = String.format("/session/%s/se/vnc", other.getId());
      toUse = new PersistentCapabilities(toUse).setCapability("se:vnc", rewrite(vncPath));
    }

    return new Session(other.getId(), externalUri, other.getStereotype(), toUse, Instant.now());
  }

  private URI rewrite(String path) {
    try {
      String scheme = "https".equals(gridUri.getScheme()) ? "wss" : "ws";
      path = NodeOptions.normalizeSubPath(gridUri.getPath()) + path;
      return new URI(
          scheme, gridUri.getUserInfo(), gridUri.getHost(), gridUri.getPort(), path, null, null);
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public NodeStatus getStatus() {
    Set<Slot> slots =
        factories.stream()
            .map(
                slot -> {
                  Instant lastStarted = Instant.EPOCH;
                  Session session = null;
                  if (!slot.isAvailable()) {
                    ActiveSession activeSession = slot.getSession();
                    if (activeSession != null) {
                      lastStarted = activeSession.getStartTime();
                      session =
                          new Session(
                              activeSession.getId(),
                              activeSession.getUri(),
                              slot.getStereotype(),
                              activeSession.getCapabilities(),
                              activeSession.getStartTime());
                    }
                  }

                  return new Slot(
                      new SlotId(getId(), slot.getId()),
                      slot.getStereotype(),
                      lastStarted,
                      session);
                })
            .collect(toUnmodifiableSet());

    Availability availability = isDraining() ? DRAINING : UP;

    // Check if consecutive session creation failures have exceeded the threshold
    if (nodeDownFailureThreshold > 0
        && consecutiveSessionFailures.get() >= nodeDownFailureThreshold) {
      availability = DOWN;
    }

    // Check status in case this Node is a RelayNode
    Optional<SessionSlot> relaySlot =
        factories.stream().filter(SessionSlot::hasRelayFactory).findFirst();
    if (relaySlot.isPresent() && !relaySlot.get().isRelayServiceUp()) {
      availability = DOWN;
    }

    return new NodeStatus(
        getId(),
        externalUri,
        maxSessionCount,
        slots,
        availability,
        heartbeatPeriod,
        getSessionTimeout(),
        getNodeVersion(),
        getOsInfo());
  }

  @Override
  public HealthCheck getHealthCheck() {
    return healthCheck;
  }

  @Override
  public void drain() {
    try (Span span = tracer.getCurrentContext().createSpan("node.drain")) {
      AttributeMap attributeMap = tracer.createAttributeMap();
      attributeMap.put(AttributeKey.LOGGER_CLASS.getKey(), getClass().getName());
      bus.fire(new NodeDrainStarted(getId()));
      draining.set(true);
      // Ensure the pendingSessions counter will not be decremented by timed out sessions not
      // included
      // in the currentSessionCount and the NodeDrainComplete will be raised to early.
      currentSessions.cleanUp();
      int currentSessionCount = getCurrentSessionCount();
      attributeMap.put("current.session.count", currentSessionCount);
      attributeMap.put("node.id", getId().toString());
      attributeMap.put("node.drain_after_session_count", this.configuredSessionCount);
      if (currentSessionCount == 0) {
        LOG.info("Firing node drain complete message");
        bus.fire(new NodeDrainComplete(getId()));
        span.addEvent("Node drain complete", attributeMap);
      } else {
        pendingSessions.set(currentSessionCount);
        span.addEvent(String.format("%s session(s) pending before draining Node", attributeMap));
      }
    }
  }

  private void checkSessionCount() {
    if (this.drainAfterSessions) {
      Lock lock = drainLock.writeLock();
      if (!lock.tryLock()) {
        // in case we can't get a write lock another thread does hold a read lock and will call
        // checkSessionCount as soon as he releases the read lock. So we do not need to wait here
        // for the other session to start and release the lock, just continue and let the other
        // session start to drain the node.
        return;
      }
      try {
        int remainingSessions = this.sessionCount.get();
        if (remainingSessions <= 0) {
          LOG.info(
              String.format(
                  "Draining Node, configured sessions value (%s) has been reached.",
                  this.configuredSessionCount));
          drain();
        }
      } finally {
        lock.unlock();
      }
    }
  }

  private boolean decrementSessionCount() {
    if (this.drainAfterSessions) {
      int remainingSessions = this.sessionCount.decrementAndGet();
      LOG.log(
          Debug.getDebugLogLevel(),
          "{0} remaining sessions before draining Node",
          remainingSessions);
      return remainingSessions >= 0;
    }
    return true;
  }

  /**
   * Restores the session count when session creation fails after decrementSessionCount() was
   * called. This prevents the node from draining prematurely due to failed session attempts.
   */
  private void restoreSessionCount() {
    if (this.drainAfterSessions) {
      int remainingSessions = this.sessionCount.incrementAndGet();
      LOG.log(
          Debug.getDebugLogLevel(),
          "Session creation failed, restored count. {0} remaining sessions before draining Node",
          remainingSessions);
    }
  }

  private Map<String, Object> toJson() {
    return Map.of(
        "id",
        getId(),
        "uri",
        externalUri,
        "maxSessions",
        maxSessionCount,
        "draining",
        isDraining(),
        "capabilities",
        factories.stream().map(SessionSlot::getStereotype).collect(Collectors.toSet()));
  }

  public static class Builder {

    private final Tracer tracer;
    private final EventBus bus;
    private final URI uri;
    private final URI gridUri;
    private final Secret registrationSecret;
    private final List<SessionSlot> factories;
    private int maxSessions = NodeOptions.DEFAULT_MAX_SESSIONS;
    private int drainAfterSessionCount = NodeOptions.DEFAULT_DRAIN_AFTER_SESSION_COUNT;
    private boolean cdpEnabled = NodeOptions.DEFAULT_ENABLE_CDP;
    private boolean bidiEnabled = NodeOptions.DEFAULT_ENABLE_BIDI;
    private Ticker ticker = Ticker.systemTicker();
    private Duration sessionTimeout = Duration.ofSeconds(NodeOptions.DEFAULT_SESSION_TIMEOUT);
    private HealthCheck healthCheck;
    private Duration heartbeatPeriod = Duration.ofSeconds(NodeOptions.DEFAULT_HEARTBEAT_PERIOD);
    private boolean managedDownloadsEnabled = false;
    private int connectionLimitPerSession = -1;
    // Default 0 means disabled (unlimited retries allowed)
    private int nodeDownFailureThreshold = 0;

    private Builder(Tracer tracer, EventBus bus, URI uri, URI gridUri, Secret registrationSecret) {
      this.tracer = Require.nonNull("Tracer", tracer);
      this.bus = Require.nonNull("Event bus", bus);
      this.uri = Require.nonNull("Remote node URI", uri);
      this.gridUri = Require.nonNull("Grid URI", gridUri);
      this.registrationSecret = Require.nonNull("Registration secret", registrationSecret);
      this.factories = new ArrayList<>();
    }

    public Builder add(Capabilities stereotype, SessionFactory factory) {
      Require.nonNull("Capabilities", stereotype);
      Require.nonNull("Session factory", factory);

      factories.add(new SessionSlot(bus, stereotype, factory));

      return this;
    }

    public Builder maximumConcurrentSessions(int maxCount) {
      this.maxSessions = Require.positive("Max session count", maxCount);
      return this;
    }

    public Builder drainAfterSessionCount(int sessionCount) {
      this.drainAfterSessionCount = sessionCount;
      return this;
    }

    public Builder enableCdp(boolean cdpEnabled) {
      this.cdpEnabled = cdpEnabled;
      return this;
    }

    public Builder enableBiDi(boolean bidiEnabled) {
      this.bidiEnabled = bidiEnabled;
      return this;
    }

    public Builder sessionTimeout(Duration timeout) {
      sessionTimeout = timeout;
      return this;
    }

    public Builder heartbeatPeriod(Duration heartbeatPeriod) {
      this.heartbeatPeriod = heartbeatPeriod;
      return this;
    }

    public Builder enableManagedDownloads(boolean enable) {
      this.managedDownloadsEnabled = enable;
      return this;
    }

    public Builder connectionLimitPerSession(int connectionLimitPerSession) {
      this.connectionLimitPerSession = connectionLimitPerSession;
      return this;
    }

    /**
     * Sets the maximum number of consecutive session creation failures allowed before the node is
     * marked as DOWN. This helps detect and isolate unhealthy nodes that consistently fail to
     * create sessions.
     *
     * @param threshold the maximum number of consecutive failures allowed (0 to disable, which is
     *     the default)
     * @return this builder
     */
    public Builder nodeDownFailureThreshold(int threshold) {
      this.nodeDownFailureThreshold = threshold;
      return this;
    }

    public LocalNode build() {
      return new LocalNode(
          tracer,
          bus,
          uri,
          gridUri,
          healthCheck,
          maxSessions,
          drainAfterSessionCount,
          cdpEnabled,
          bidiEnabled,
          ticker,
          sessionTimeout,
          heartbeatPeriod,
          List.copyOf(factories),
          registrationSecret,
          managedDownloadsEnabled,
          connectionLimitPerSession,
          nodeDownFailureThreshold);
    }

    public Advanced advanced() {
      return new Advanced();
    }

    public class Advanced {

      public Advanced clock(Clock clock) {
        ticker = () -> clock.instant().toEpochMilli() * Duration.ofMillis(1).toNanos();
        return this;
      }

      public Advanced healthCheck(HealthCheck healthCheck) {
        Builder.this.healthCheck = Require.nonNull("Health check", healthCheck);
        return this;
      }

      public Node build() {
        return Builder.this.build();
      }
    }
  }
}
