#include "src/database/internal/DatabaseMessageCursor.h"

#include "src/database/internal/InternalDatabaseInterface.h"
#include "src/exceptions/InternalErrorException.h"
#include "src/utility/Logging.h"

#include <QVariant>

namespace openmittsu {
	namespace database {
		namespace internal {

			using namespace openmittsu::dataproviders::messages;

			DatabaseMessageCursor::DatabaseMessageCursor(InternalDatabaseInterface* database) : m_database(database), m_messageId(0), m_isMessageIdValid(false), m_messageType() {
				//
			}

			DatabaseMessageCursor::~DatabaseMessageCursor() {
				//
			}

			bool DatabaseMessageCursor::isValid() const {
				return m_isMessageIdValid;
			}

			bool DatabaseMessageCursor::seekToFirst() {
				return getFirstOrLastMessageId(true);
			}

			bool DatabaseMessageCursor::seekToLast() {
				return getFirstOrLastMessageId(false);
			}

			bool DatabaseMessageCursor::seek(openmittsu::protocol::MessageId const& messageId) {
				QSqlQuery query(m_database->getQueryObject());
				QString const queryString = QStringLiteral("SELECT `apiid`, `uid`, `sort_by`, `%3` AS `messageType` FROM `%1` WHERE %2 AND `apiid` = :apiid;").arg(getTableName()).arg(getWhereString()).arg(getMessageTypeField());
				if (!query.prepare(queryString)) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not prepare message seek query. SQL error: " << query.lastError().text().toStdString();
				}
				query.bindValue(QStringLiteral(":apiid"), QVariant(messageId.toQString()));
				bindWhereStringValues(query);

				if (!query.exec() || !query.isSelect()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message seek query for table " << getTableName().toStdString() << " for message ID \"" << messageId.toString() << "\". Query error: " << query.lastError().text().toStdString();
				}

				if (query.next()) {
					m_isMessageIdValid = true;
					m_messageId = openmittsu::protocol::MessageId(query.value(QStringLiteral("apiid")).toString());
					m_uid = query.value(QStringLiteral("uid")).toString();
					m_sortByValue = query.value(QStringLiteral("sort_by")).toLongLong();
					m_messageType = query.value(QStringLiteral("messageType")).toString();
					return true;
				} else {
					m_isMessageIdValid = false;
					return false;
				}
			}

			bool DatabaseMessageCursor::seekByUuid(QString const& uuid) {
				QSqlQuery query(m_database->getQueryObject());
				QString const queryString = QStringLiteral("SELECT `apiid`, `uid`, `sort_by`, `%3` AS `messageType` FROM `%1` WHERE %2 AND `uid` = :uid;").arg(getTableName()).arg(getWhereString()).arg(getMessageTypeField());
				if (!query.prepare(queryString)) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not prepare message seekByUuid query. SQL error: " << query.lastError().text().toStdString();
				}
				query.bindValue(QStringLiteral(":uid"), QVariant(uuid));
				bindWhereStringValues(query);

				if (!query.exec() || !query.isSelect()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message seek query for table " << getTableName().toStdString() << " for UUID \"" << uuid.toStdString() << "\". Query error: " << query.lastError().text().toStdString();
				}

				if (query.next()) {
					m_isMessageIdValid = true;
					m_messageId = openmittsu::protocol::MessageId(query.value(QStringLiteral("apiid")).toString());
					m_uid = query.value(QStringLiteral("uid")).toString();
					m_sortByValue = query.value(QStringLiteral("sort_by")).toLongLong();
					m_messageType = query.value(QStringLiteral("messageType")).toString();
					return true;
				} else {
					m_isMessageIdValid = false;
					return false;
				}
			}

			bool DatabaseMessageCursor::getFollowingMessageId(bool ascending) {
				if (!m_isMessageIdValid) {
					return false;
				}

				QString sortOrder;
				QString sortOrderSign;
				if (ascending) {
					sortOrder = QStringLiteral("ASC");
					sortOrderSign = QStringLiteral(">");
				} else {
					sortOrder = QStringLiteral("DESC");
					sortOrderSign = QStringLiteral("<");
				}

#if defined(QT_VERSION) && (QT_VERSION >= QT_VERSION_CHECK(5, 10, 0)) && (QT_VERSION < QT_VERSION_CHECK(5, 10, 1))
				// Check in two steps to mitigate a cool bug in the query engine.
				QSqlQuery query(m_database->getQueryObject());
				query.prepare(QStringLiteral("SELECT `apiid`, `uid`, `sort_by`, `%5` AS `messageType` FROM `%1` WHERE (%2) AND (((`sort_by` = :sortByValue) AND (`uid` %3 :uid))) ORDER BY `sort_by` %4, `uid` %4 LIMIT 1;").arg(getTableName()).arg(getWhereString()).arg(sortOrderSign).arg(sortOrder).arg(getMessageTypeField()));
				bindWhereStringValues(query);
				query.bindValue(QStringLiteral(":sortByValue"), QVariant(m_sortByValue));
				query.bindValue(QStringLiteral(":uid"), QVariant(m_uid));

				if (!query.exec() || !query.isSelect()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message iteration query for table " << getTableName().toStdString() << ". Query error: " << query.lastError().text().toStdString();
				}

				if (query.next()) {
					m_isMessageIdValid = true;
					m_messageId = openmittsu::protocol::MessageId(query.value(QStringLiteral("apiid")).toString());
					m_uid = query.value(QStringLiteral("uid")).toString();
					m_sortByValue = query.value(QStringLiteral("sort_by")).toLongLong();
					m_messageType = query.value(QStringLiteral("messageType")).toString();
					return true;
				} else {
					query.prepare(QStringLiteral("SELECT `apiid`, `uid`, `sort_by`, `%5` AS `messageType` FROM `%1` WHERE (%2) AND ((`sort_by` %3 :sortByValue)) ORDER BY `sort_by` %4, `uid` %4 LIMIT 1;").arg(getTableName()).arg(getWhereString()).arg(sortOrderSign).arg(sortOrder).arg(getMessageTypeField()));
					bindWhereStringValues(query);
					query.bindValue(QStringLiteral(":sortByValue"), QVariant(m_sortByValue));

					if (!query.exec() || !query.isSelect()) {
						throw openmittsu::exceptions::InternalErrorException() << "Could not execute message iteration query for table " << getTableName().toStdString() << ". Query error: " << query.lastError().text().toStdString();
					}

					if (query.next()) {
						m_isMessageIdValid = true;
						m_messageId = openmittsu::protocol::MessageId(query.value(QStringLiteral("apiid")).toString());
						m_uid = query.value(QStringLiteral("uid")).toString();
						m_sortByValue = query.value(QStringLiteral("sort_by")).toLongLong();
						m_messageType = query.value(QStringLiteral("messageType")).toString();
						return true;
					} else {
						return false;
					}
				}
#else
				QSqlQuery query(m_database->getQueryObject());
				query.prepare(QStringLiteral("SELECT `apiid`, `uid`, `sort_by`, `%5` AS `messageType` FROM `%1` WHERE %2 AND ((`sort_by` %3 :sortByValue) OR ((`sort_by` = :sortByValue) AND (`uid` %3 :uid))) ORDER BY `sort_by` %4, `uid` %4 LIMIT 1;").arg(getTableName()).arg(getWhereString()).arg(sortOrderSign).arg(sortOrder).arg(getMessageTypeField()));
				bindWhereStringValues(query);
				query.bindValue(QStringLiteral(":sortByValue"), QVariant(m_sortByValue));
				query.bindValue(QStringLiteral(":uid"), QVariant(m_uid));

				if (!query.exec() || !query.isSelect()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message iteration query for table " << getTableName().toStdString() << ". Query error: " << query.lastError().text().toStdString();
				}

				if (query.next()) {
					m_isMessageIdValid = true;
					m_messageId = openmittsu::protocol::MessageId(query.value(QStringLiteral("apiid")).toString());
					m_uid = query.value(QStringLiteral("uid")).toString();
					m_sortByValue = query.value(QStringLiteral("sort_by")).toLongLong();
					m_messageType = query.value(QStringLiteral("messageType")).toString();
					return true;
				} else {
					return false;
				}
#endif
			}

			bool DatabaseMessageCursor::getFirstOrLastMessageId(bool first) {
				QString sortOrder;
				if (first) {
					sortOrder = QStringLiteral("ASC");
				} else {
					sortOrder = QStringLiteral("DESC");
				}

				QSqlQuery query(m_database->getQueryObject());
				QString const queryString = QStringLiteral("SELECT `apiid`, `uid`, `sort_by`, `%4` AS `messageType` FROM `%1` WHERE %2 ORDER BY `sort_by` %3, `uid` %3 LIMIT 1;").arg(getTableName()).arg(getWhereString()).arg(sortOrder).arg(getMessageTypeField());
				if (!query.prepare(queryString)) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not prepare message firstOrLastMessageId query. SQL error: " << query.lastError().text().toStdString();
				}
				bindWhereStringValues(query);

				if (!query.exec() || !query.isSelect()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message first/last query for table " << getTableName().toStdString() << ". Query error: " << query.lastError().text().toStdString();
				}

				if (query.next()) {
					m_isMessageIdValid = true;
					m_messageId = openmittsu::protocol::MessageId(query.value(QStringLiteral("apiid")).toString());
					m_uid = query.value(QStringLiteral("uid")).toString();
					m_sortByValue = query.value(QStringLiteral("sort_by")).toLongLong();
					m_messageType = query.value(QStringLiteral("messageType")).toString();
					return true;
				} else {
					return false;
				}
			}

			QVector<QString> DatabaseMessageCursor::getLastMessages(std::size_t n) const {
				QSqlQuery query(m_database->getQueryObject());
				query.prepare(QStringLiteral("SELECT `uid` FROM `%1` WHERE %2 ORDER BY `sort_by` DESC, `uid` DESC LIMIT %3;").arg(getTableName()).arg(getWhereString()).arg(n));
				bindWhereStringValues(query);

				if (!query.exec() || !query.isSelect()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message enumeration query for table " << getTableName().toStdString() << ". Query error: " << query.lastError().text().toStdString();
				}

				QVector<QString> result;

				while (query.next()) {
					result.push_back(query.value(QStringLiteral("uid")).toString());
				}

				return result;
			}

			void DatabaseMessageCursor::deleteMessage(bool doAnnounce) {
				if (!m_isMessageIdValid) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message deletion query for invalid message!";
				}

				getDatabase()->removeAllMediaItems(getMessageUuid());

				QSqlQuery query(getDatabase()->getQueryObject());
				query.prepare(QStringLiteral("DELETE FROM `%1` WHERE %2 AND `uid` = :uid;").arg(getTableName()).arg(getWhereString()));
				bindWhereStringValues(query);
				query.bindValue(QStringLiteral(":uid"), QVariant(getMessageUuid()));
				if (!query.exec()) {
					throw openmittsu::exceptions::InternalErrorException() << "Could not execute message deletion query for table " << getTableName().toStdString() << ". Query error: " << query.lastError().text().toStdString();
				}
				if (doAnnounce) {
					getDatabase()->announceMessageDeleted(getMessageUuid());
				}
				m_isMessageIdValid = false;
			}

			bool DatabaseMessageCursor::next() {
				return getFollowingMessageId(true);
			}

			bool DatabaseMessageCursor::previous() {
				return getFollowingMessageId(false);
			}

			InternalDatabaseInterface* DatabaseMessageCursor::getDatabase() const {
				return m_database;
			}

			openmittsu::protocol::MessageId const& DatabaseMessageCursor::getMessageId() const {
				return m_messageId;
			}

			QString const& DatabaseMessageCursor::getMessageUuid() const {
				return m_uid;
			}

		}
	}
}
