// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "mq3.h"
#include "collection.h"
#include "collectionview.h"
#include "constants.h"
#include "organizer.h"
#include "configuration.h"
#include "controls.h"
#include "scrolllabel.h"
#include "equalizer.h"
#include "mainvisual.h"
#include "mediasearch.h"
#include "mediaview.h"
#include "icons.h"
#include "song.h"

#include "streaminput.h"
#include "output.h"
#include "decoder.h"

#include <qapplication.h>
#include <qaccel.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qfont.h>
#include <qframe.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qfile.h>
#include <qsettings.h>
#include <qslider.h>
#include <qstyle.h>
#include <qtooltip.h>

// for now... use Factories later
#include "audiooutput.h"

static MQ3 *INSTANCE = 0;


MQ3::MQ3(void)
    : QWidget(0, "mq3"), input(0), output(0), decoder(0), source(0),
      firstShow(false), playOnStartup(false), showOrganizerOnStartup(false),
      showEqualizerOnStartup(false), remainingTime(false), seeking(false),
      enableEq(false), outputBufferSize(0)
{
    INSTANCE = this;

    setCaption(tr("MQ3"));

    // create player
    mainframe = new QFrame(this);
    mainframe->setFrameStyle( QFrame::Sunken | QFrame::Box );

    mainvisual = new MainVisual(mainframe);

    timeLabel = new QLabel(mainframe);
    QFont fn(timeLabel->font());
    fn.setPointSize(fn.pointSize() * 2);
    fn.setBold(TRUE);
    timeLabel->setFont(fn);
    timeLabel->setAlignment(AlignCenter);
    QFontMetrics fm(fn);
    QSize s(fm.width("MMm MMs"), fm.height());
    timeLabel->setMinimumSize(s);
    timeLabel->setText("--m --s");
    timeLabel->setSizePolicy(QSizePolicy(QSizePolicy::Minimum,
					 QSizePolicy::Minimum));

    seeklabel = new QLabel(tr("Seek"), mainframe);

    seekbar = new QSlider(Qt::Horizontal, mainframe);
    seekbar->setFocusPolicy(NoFocus);
    seekbar->setTracking(false);
    connect(seekbar, SIGNAL(sliderPressed()), SLOT(startseek()));
    connect(seekbar, SIGNAL(sliderReleased()), SLOT(doneseek()));
    connect(seekbar, SIGNAL(valueChanged(int)), SLOT(seek(int)));

    songLabel = new ScrollLabel(mainframe);
    songLabel->setText(tr("No song loaded."));
    songLabel->setSizePolicy(QSizePolicy(QSizePolicy::Minimum,
					 QSizePolicy::Fixed));

    statusLabel = new ScrollLabel(mainframe);
    statusLabel->setText(tr("Welcome to %1").arg(VERSION));
    statusLabel->setSizePolicy(QSizePolicy(QSizePolicy::Minimum,
					   QSizePolicy::Fixed));

    controls = new Controls(this);

    QGridLayout *grid1 = new QGridLayout(2, 2);
    grid1->addMultiCellWidget(timeLabel, 0, 0, 0, 1);
    grid1->addWidget(seeklabel, 1, 0);
    grid1->addWidget(seekbar, 1, 1);

    QHBoxLayout *hbox1 = new QHBoxLayout(6);
    hbox1->addWidget(mainvisual);
    hbox1->addLayout(grid1);

    QVBoxLayout *vbox2 = new QVBoxLayout(mainframe, 2, 2);
    vbox2->addWidget(songLabel);
    vbox2->addLayout(hbox1);
    vbox2->addWidget(statusLabel);

    QVBoxLayout *vbox3 = new QVBoxLayout(this, 2, 2);
    vbox3->addWidget(mainframe);
    vbox3->addWidget(controls);

    setFixedSize(minimumSizeHint());

    // create dialogs/toplevels
    config = new Configuration(this);
    organizer = new Organizer();
    equalizer = new Equalizer();

    // connect!

    connect(organizer, SIGNAL(Previous()), SLOT(previous()));
    connect(organizer, SIGNAL(Stop()), SLOT(stop()));
    connect(organizer, SIGNAL(Play()), SLOT(play()));
    connect(organizer, SIGNAL(Pause()), SLOT(pause()));
    connect(organizer, SIGNAL(Next()), SLOT(next()));
    connect(organizer, SIGNAL(songSelected()), SLOT(changeSong()));

    connect(config, SIGNAL(configChanged()), SLOT(configChanged()));

    connect(equalizer, SIGNAL(presetChanged(const EqPreset &)),
	    SLOT(eqChanged(const EqPreset &)));

    connect(controls, SIGNAL(Play()), SLOT(play()));
    connect(controls, SIGNAL(Previous()), SLOT(previous()));
    connect(controls, SIGNAL(Next()), SLOT(next()));
    connect(controls, SIGNAL(Stop()), SLOT(stop()));
    connect(controls, SIGNAL(Pause()), SLOT(pause()));
    connect(controls, SIGNAL(Organizer()), organizer, SLOT(show()));
    connect(controls, SIGNAL(Equalizer()), equalizer, SLOT(show()));
    connect(controls, SIGNAL(Configuration()), config, SLOT(show()));

    config->init();
}


MQ3::~MQ3(void)
{
    stopAll();

    delete organizer;
    organizer = 0;
    delete equalizer;
    equalizer = 0;
    delete config;
    config = 0;

    IconLoader::cleanup();
}


void MQ3::configChanged()
{
    QIconSet iconset = IconLoader::load("mq3");
    setIcon(iconset.pixmap(QIconSet::Large, QIconSet::Normal));

    QSettings settings;
    QString val;

    val = settings.readEntry( "/MQ3/Visual/backColor" );
    if ( ! val.isEmpty() ) {
	mainframe->setPalette( QColor( val ) );
	// the seekbar can get hard to see, so keep it with the default palette
	QPalette spal( palette() ), fpal( mainframe->palette() );
	spal.setColor( QPalette::Active, QColorGroup::Background,
		       fpal.color( QPalette::Active, QColorGroup::Background ) );
	spal.setColor( QPalette::Inactive, QColorGroup::Background,
		       fpal.color( QPalette::Active, QColorGroup::Background ) );
	spal.setColor( QPalette::Disabled, QColorGroup::Background,
		       fpal.color( QPalette::Active, QColorGroup::Background ) );
	seekbar->setPalette( spal );
    }

    val = settings.readEntry( "/MQ3/Visual/frameShape" );
    if ( ! val.isEmpty() ) {
	if ( val != "None" ) {
	    int fshape = QFrame::NoFrame, fstyle = QFrame::NoFrame;
	    int lw = 1;

	    if ( val == "Panel" ) {
		fshape = QFrame::StyledPanel;
		lw = style().pixelMetric( QStyle::PM_DefaultFrameWidth );
	    } else if (val == "Boxed Panel" ) {
		fshape = QFrame::Box;
		lw = 1;
	    } else { // shouldn't happen!
		fshape = QFrame::Box;
		lw = 1;
	    }

	    val = settings.readEntry( "/MQ3/Visual/frameStyle" );
	    if ( ! val.isEmpty() ) {
		if ( val == "Raised" )
		    fstyle = QFrame::Raised;
		else if (val == "Sunken" )
		    fstyle = QFrame::Sunken;
		else // shouldn't happen!
		    fstyle = QFrame::Sunken;
	    }

	    mainframe->setFrameShape( QFrame::Shape( fshape ) );
	    mainframe->setFrameShadow( QFrame::Shadow( fstyle ) );
	    mainframe->setLineWidth( lw );
	} else
	    mainframe->setFrameShape( QFrame::NoFrame );
    }

    controls->configChanged(settings);
    equalizer->configChanged(settings);
    organizer->configChanged(settings);
    mainvisual->configChanged(settings);

    showOrganizerOnStartup =
	settings.readBoolEntry("/MQ3/Organizer/showOnStartup", false);
    showEqualizerOnStartup =
	settings.readBoolEntry("/MQ3/Equalizer/showOnStartup", false);

    outputBufferSize = settings.readNumEntry("/MQ3/Output/bufferSize", 128);
    remainingTime = settings.readBoolEntry("/MQ3/showRemainingTime", false);
    playOnStartup = settings.readBoolEntry("/MQ3/playOnStartup", false);
    enableEq = settings.readBoolEntry("/MQ3/Equalizer/enabled", false);

    int ss = settings.readNumEntry("/MQ3/scrollSpeed", 50);
    int sd = settings.readNumEntry("/MQ3/scrollStopDelay", 50);
    statusLabel->setScrollSpeed(ss);
    statusLabel->setStopDelay(sd);
    songLabel->setScrollSpeed(ss);
    songLabel->setStopDelay(sd);
}


void MQ3::play(void)
{
    stop();

    if (! CollectionView::instance()->currentCollection())
	return;

    if (! source)
	source = CollectionView::instance()->currentCollection()->currentSong;

    if (! source &&
	MediaView::instance()->selectedItem() &&
	MediaView::instance()->selectedItem()->rtti() == Song::RTTI)
	source = (Song *) MediaView::instance()->selectedItem();

    if (! source)
	source = CollectionView::instance()->currentCollection()->next();

    if (MediaView::instance()->selectedItem() &&
	MediaView::instance()->selectedItem()->rtti() == Song::RTTI &&
	(Song *) MediaView::instance()->selectedItem() != source) {
	source = (Song *) MediaView::instance()->selectedItem();
	CollectionView::instance()->currentCollection()->setCurrentSong(source);
    }

    if (! source)
	return;

    QUrl sourceurl(source->location());
    QString sourcename(source->location());
    QString sourceeq(source->equalizer());
    source = 0;

    if (output)
	fprintf(stderr, "output not deleted!\n");

    output = new AudioOutput(outputBufferSize * 1024, "/dev/dsp");
    output->setBufferSize(outputBufferSize * 1024);
    output->addListener(this);
    output->addListener(mainvisual);
    output->addVisual(mainvisual);

    if (! output->initialize())
	return;

    if (! sourceurl.isLocalFile()) {
	StreamInput streaminput(sourceurl);
	streaminput.setup();
	input = streaminput.socket();
    } else
	input = new QFile(sourceurl.toString(FALSE, FALSE));

    if (decoder && ! decoder->factory()->supports(sourcename))
	decoder = 0;

    if (! decoder) {
	decoder = Decoder::create(sourcename, input, output);

	if (! decoder) {
	    printf("MQ3: unsupported fileformat\n");
	    stopAll();
	    return;
	}

	decoder->setBlockSize(globalBlockSize);
	decoder->addListener(this);
    } else {
	decoder->setInput(input);
	decoder->setOutput(output);
    }

    mainvisual->setDecoder(decoder);
    mainvisual->setOutput(output);

    if (decoder->isEQSupported() && ! sourceeq.isNull()) {
	decoder->setEQEnabled(enableEq);
	decoder->setEQ(equalizer->preset(sourceeq));
    } else
	// clear any previous eq setting
	decoder->setEQEnabled(false);

    if (decoder->initialize()) {
	seekbar->blockSignals(true);
	seekbar->setMinValue(0);
	seekbar->setValue(0);
	seekbar->setMaxValue(int(decoder->lengthInSeconds()));
	if (seekbar->maxValue() == 0) {
	    seeklabel->setEnabled(false);
	    seekbar->setEnabled(false);
	} else {
	    seeklabel->setEnabled(true);
	    seekbar->setEnabled(true);
	}
	seekbar->blockSignals(false);

	output->start();
	decoder->start();
    }
}


void MQ3::pause(void)
{
    if (output) {
	output->mutex()->lock();
	output->pause();
	output->mutex()->unlock();
    }

    // wake up threads
    if (decoder) {
	decoder->mutex()->lock();
	decoder->cond()->wakeAll();
	decoder->mutex()->unlock();
    }

    if (output) {
	output->recycler()->mutex()->lock();
	output->recycler()->cond()->wakeAll();
	output->recycler()->mutex()->unlock();
    }
}


void MQ3::stop(void)
{
    if (decoder && decoder->running()) {
	decoder->mutex()->lock();
	decoder->stop();
	decoder->mutex()->unlock();
    }

    if (output && output->running()) {
	output->mutex()->lock();
	output->stop();
	output->mutex()->unlock();
    }

    // wake up threads
    if (decoder) {
	decoder->mutex()->lock();
	decoder->cond()->wakeAll();
	decoder->mutex()->unlock();
    }

    if (output) {
	output->recycler()->mutex()->lock();
	output->recycler()->cond()->wakeAll();
	output->recycler()->mutex()->unlock();
    }

    if (decoder)
	decoder->wait();

    if (output)
	output->wait();

    mainvisual->setDecoder(0);
    mainvisual->setOutput(0);

    delete output;
    output = 0;
    delete input;
    input = 0;
}


void MQ3::stopAll()
{
    stop();

    if (decoder) {
	decoder->removeListener(this);
	decoder = 0;
    }
}


void MQ3::previous()
{
    stop();
    source = 0;
    if (! CollectionView::instance()->currentCollection())
	return;
    source = CollectionView::instance()->currentCollection()->previous();
    play();
}


void MQ3::next()
{
    stop();
    source = 0;
    if (! CollectionView::instance()->currentCollection())
	return;
    source = CollectionView::instance()->currentCollection()->next();
    play();
}


void MQ3::startseek()
{
    seeking = true;
}


void MQ3::doneseek()
{
    seeking = false;
}


void MQ3::seek(int pos)
{
    if (output && output->running()) {
	output->mutex()->lock();
	output->seek(pos);

	if (decoder && decoder->running()) {
	    decoder->mutex()->lock();
	    decoder->seek(pos);

	    if (mainvisual) {
		mainvisual->mutex()->lock();
		mainvisual->prepare();
		mainvisual->mutex()->unlock();
	    }

	    decoder->mutex()->unlock();
	}

	output->mutex()->unlock();
    }
}


void MQ3::changeSong()
{
    stop();
    source = 0;
    if (! CollectionView::instance()->currentCollection())
	return;
    if (MediaView::instance()->selectedItem() &&
	MediaView::instance()->selectedItem()->rtti() == Song::RTTI)
	source = (Song *) MediaView::instance()->selectedItem();
    play();
}

void MQ3::eqChanged(const EqPreset &preset)
{
    if (! decoder)
	return;

    decoder->mutex()->lock();
    if (decoder->isEQSupported() &&
	decoder->EQ().name() == preset.name())
	decoder->setEQ(preset);
    decoder->mutex()->unlock();
}

void MQ3::closeEvent(QCloseEvent *event)
{
    stopAll();

    hide();

    organizer->deinit();
    equalizer->deinit();

    organizer->hide();
    equalizer->hide();

    event->accept();
}


void MQ3::showEvent(QShowEvent *event)
{
    if (!firstShow) {
	organizer->load();
	if (playOnStartup)
	    play();
	if (showOrganizerOnStartup)
	    organizer->show();
	if (showEqualizerOnStartup)
	    equalizer->show();
    }

    firstShow = TRUE;

    QWidget::showEvent(event);
}


void MQ3::customEvent(QCustomEvent *event)
{
    switch ((int) event->type()) {
    case OutputEvent::Playing:
	{
	    statusString = tr("Playing stream.");

	    MediaSearch::instance()->clear();

	    statusLabel->setText(statusString + "  " + infoString);
	    QString title =
		CollectionView::instance()->currentCollection()->currentSong->title();
	    songLabel->setText(title);

	    break;
	}

    case OutputEvent::Buffering:
	{
	    statusString = tr("Buffering stream.");

	    MediaSearch::instance()->clear();
	    statusLabel->setText(statusString);

	    break;
	}

    case OutputEvent::Paused:
	{
	    statusString = tr("Stream paused.");

	    statusLabel->setText(statusString + "  " + infoString);

	    break;
	}

    case OutputEvent::Info:
	{
	    OutputEvent *oe = (OutputEvent *) event;

	    int em, es, rs;

	    if (remainingTime && seekbar->maxValue() > 0) {
		rs = seekbar->maxValue() - oe->elapsedSeconds();
		if (rs < 0)
		    rs = 0;
	    } else
		rs = oe->elapsedSeconds();

	    em = rs / 60;
	    es = rs % 60;

	    timeString.sprintf("%02dm %02ds", em, es);
	    timeLabel->setText(timeString);

	    if (! seeking) {
		seekbar->blockSignals(true);
		seekbar->setValue(oe->elapsedSeconds());
		seekbar->blockSignals(false);
	    }

	    QToolTip::add(seekbar,
			  QString(tr("Elapsed time: %1m %2s of %3m %4s")).
			  arg(em).arg(es).arg(seekbar->maxValue() / 60).
			  arg(seekbar->maxValue() % 60));

	    infoString.sprintf("%d kbps, %.1f kHz %s.",
			       oe->bitrate(), float(oe->frequency()) / 1000.0,
			       oe->channels() > 1 ? "stereo" : "mono");
	    statusLabel->setText(statusString + "  " + infoString);

	    break;
	}

    case OutputEvent::Error:
	{
	    statusString = tr("Output error.");
	    timeString = "--m --s";

	    statusLabel->setText(statusString + "  " + infoString);
	    timeLabel->setText(timeString);

	    OutputEvent *aoe = (OutputEvent *) event;
	    QMessageBox::critical(qApp->activeWindow(),
				  statusString,
				  *aoe->errorMessage());

	    stopAll();

	    break;
	}

    case DecoderEvent::Stopped:
	{
	    statusString = tr("Stream stopped.");
	    timeString = "--m --s";

	    statusLabel->setText(statusString + "  " + infoString);
	    timeLabel->setText(timeString);

	    break;
	}

    case DecoderEvent::Finished:
	{
	    statusString = tr("Finished playing stream.");

	    statusLabel->setText(statusString + "  " + infoString);

	    if (organizer->playMode() != Organizer::OneSong)
		next();

	    break;
	}

    case DecoderEvent::Error:
	{
	    stopAll();
	    QApplication::sendPostedEvents();

	    statusString = tr("Decoder error.");

	    statusLabel->setText(statusString);

	    DecoderEvent *dxe = (DecoderEvent *) event;
	    QMessageBox::critical(qApp->activeWindow(),
				  statusString,
				  *dxe->errorMessage());
	    break;
	}
    }

    QWidget::customEvent(event);
}

MQ3 *MQ3::instance()
{
    return INSTANCE;
}
