// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "collection.h"
#include "collectionview.h"
#include "constants.h"
#include "mediaview.h"
#include "organizer.h"
#include "song.h"

#include <qdatetime.h>
#include <qpainter.h>

#include <stdlib.h>

#if defined( Q_OS_WIN32 )
#define srandom srand
#define random rand
#endif

int Collection::RTTI = 141834;

Collection::Collection(CollectionView *c, const QString &t)
    : QListViewItem(c, t), current(false), loading(false), currentSong(0)
{
    setExpandable(false);
    setSelectable(true);

    setPixmap(0, CollectionView::instance()->iconset.pixmap(QIconSet::Small,
							    QIconSet::Normal));
}

Collection::~Collection()
{
    QValueListIterator<SongState *> it = songlist.begin();
    SongState *state;

    while (it != songlist.end()) {
	state = *it++;
	delete state;
    }
}

void Collection::paintCell(QPainter *p, const QColorGroup &cg, int a, int b, int c)
{
    if (current) {
	QFont font(p->font());
	font.setBold(true);
	p->setFont(font);
    }
    QListViewItem::paintCell(p, cg, a, b, c);
}

Song *Collection::next()
{
    Song *song = 0;

    if (currentSong && currentSong->isOn()) {
	history.push(currentSong);

	while (history.count() > historySize)
	    history.remove(history.begin());
    }

    if (Organizer::instance()->playMode() == Organizer::Random && randomlist.count()) {
	QTime dt(0, 0, 0);
	::srandom(dt.msecsTo(QTime::currentTime()));
	int pos = ::random() % randomlist.count();

	song = randomlist[pos];
    } else if (Organizer::instance()->playMode() == Organizer::RepeatOne) {
	if (! currentSong) {
	    // find first song
	    QListViewItemIterator iter(MediaView::instance());
	    QListViewItem *item;

	    song = 0;
	    while (! song && (item = iter.current()) != 0) {
		iter++;

		if (item->rtti() == Song::RTTI && ((Song *) item)->isOn()) {
		    //this is a song
		    song = (Song *) item;
		}
	    }
	} else
	    song = currentSong;
    } else if (! currentSong) {
	// find first song
	QListViewItemIterator iter(MediaView::instance());
	QListViewItem *item;

	song = 0;
	while (! song && (item = iter.current()) != 0) {
	    iter++;

	    if (item->rtti() == Song::RTTI && ((Song *) item)->isOn()) {
		//this is a song
		song = (Song *) item;
	    }
	}
    } else {
	QListViewItemIterator iter(currentSong);
	iter++;
	QListViewItem *item;
	song = 0;
	while ((item = iter.current()) != 0) {
	    iter++;

	    if (item->rtti() == Song::RTTI && ((Song *) item)->isOn()) {
		song = (Song *) item;
		break;
	    }
	}

	if (! song && Organizer::instance()->playMode() == Organizer::RepeatAll) {
	    // find first song
	    QListViewItemIterator iter(MediaView::instance());
	    QListViewItem *item;

	    while (! song && (item = iter.current()) != 0) {
		iter++;

		if (item->rtti() == Song::RTTI && ((Song *) item)->isOn()) {
		    //this is a song
		    song = (Song *) item;
		}
	    }
	}
    }

    setCurrentSong(song);
    return song;
}

Song *Collection::previous()
{
    if (! history.count())
	return 0;

    Song *song = history.pop();
    setCurrentSong(song);
    return song;
}

void Collection::add(Song *song, int pri)
{
    SongState *state = new SongState(song);
    songlist.append(state);

    while (pri--)
	randomlist.append(song);
}

void Collection::remove(Song *song)
{
    QValueListIterator<SongState *> it = songlist.begin();
    SongState *state;
    while (it != songlist.end()) {
	state = *it++;

	if (song->signature() == state->signature)
	    // remove this guy...
	    songlist.remove(state);
    }

    history.remove(song);
    randomlist.remove(song);
}

Collection::SongState *Collection::match(Song *song)
{
    // see if song matches a song in songlist

    QValueListIterator<SongState *> it = songlist.begin();
    SongState *state, *ret = 0;
    while (it != songlist.end()) {
	state = *it++;

	if (song->signature() == state->signature) {
	    ret = state;
	    break;
	}
    }

    return ret;
}

void Collection::setCurrent(bool c)
{
    // Organizer::instance()->Stop();

    current = c;
    loading = true;

    QPixmap p =
	CollectionView::instance()->iconset.pixmap(QIconSet::Small,
						   (current ? QIconSet::Active :
						    QIconSet::Normal));
    setPixmap(0, p);

    randomlist.clear();
    history.clear();
    currentSong = 0;

    if (current) {
	// load
	Song *song;
	SongState *state;
	int i;

	// setup songs in our songlist
	QListViewItem *item = MediaView::instance()->firstChild();
	while (item) {
	    if (item->rtti() == Song::RTTI) {
		song = (Song *) item;

		if ((state = match(song)) != 0) {
		    // this song is in our songlist - enable it and set the priority
		    song->setOn(state->on);
		    song->setCurrent(false);
		    song->setPriority(state->priority);

		    i = state->priority;
		    while (i--)
			randomlist.append(song);
		}
	    }

	    if (item->firstChild())
		item = item->firstChild();
	    else if (item->nextSibling())
		item = item->nextSibling();
	    else {
		while (item && ! item->nextSibling())
		    item = item->parent();

		if (item)
		    item = item->nextSibling();
	    }
	}
    } else {
	// unload
	Song *song;

	// disable ALL songs
	QListViewItem *item = MediaView::instance()->firstChild();
	while (item) {
	    if (item->rtti() == Song::RTTI) {
		// this is a song
		song = (Song *) item;
		song->setOn(false);
		song->setPriority(1);
		song->setCurrent(false);
		song->repaint();
	    }

	    if (item->firstChild())
		item = item->firstChild();
	    else if (item->nextSibling())
		item = item->nextSibling();
	    else {
		while (item && ! item->nextSibling())
		    item = item->parent();

		if (item)
		    item = item->nextSibling();
	    }
	}
    }

    loading = false;
    repaint();
    MediaView::instance()->clearSelection();
}

void Collection::setCurrentSong(Song *song, bool repaint)
{
    if (currentSong) {
	currentSong->setCurrent(false);
	if (repaint)
	    currentSong->repaint();
    }
    currentSong = 0;

    if (! song)
	return;

    if (! song->isOn())
	song->setOn(true);

    currentSong = song;
    currentSong->setCurrent(true);
    if (repaint) {
	MediaView::instance()->setCurrentItem(currentSong);
	MediaView::instance()->ensureItemVisible(currentSong);
	currentSong->repaint();
    }
}

void Collection::dataDescription(QDataStream &stream) const
{
    stream << text(0) << songlist.count();

    QValueListConstIterator<SongState *> it = songlist.begin();
    SongState *state = 0;
    QString g;
    int p, o;
    while (it != songlist.end()) {
	state = *it++;

	g = state->signature.toString();
	p = state->priority;
	o = state->on;
	stream << g << p << o;
    }
}

int Collection::rtti() const
{
    return Collection::RTTI;
}
