// Copyright (c) 2000 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "mq3d.h"
#include "connection.h"
#include "collections.h"

#ifndef INSTALLDIR
#define INSTALLDIR "."
#endif

#include <qapplication.h>
#include <qsettings.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>


MQ3d::MQ3d(bool regenerate)
    : QServerSocket(mq3dListenPort), err(FALSE)
{
    if (! ok()) {
	qWarning("MQ3d: error binding/listening to port, exiting.");
	err = TRUE;
	return;
    }

    QFileInfo fi(QString(INSTALLDIR) + "/etc/mq3d");
    if (! fi.exists() || ! fi.isWritable()) {
	// make sure ~/.mq3d exists
	fi.setFile(QDir::homeDirPath() + "/.mq3d");
	if (! fi.exists())
	    QDir::home().mkdir(".mq3d");

	fi.setFile(QDir::homeDirPath() + "/.mq3d/collections");
	if (! fi.exists())
	    regenerate = TRUE;
    } else
	fi.setFile(fi.absFilePath() + "/collections");

    collpath = fi.absFilePath();

    if (regenerate)
	generate();

    colls = new Collections(collpath, this);
}


MQ3d::~MQ3d()
{
}


void MQ3d::newConnection(int socket)
{
    (void) new Connection(socket, this);
}


void MQ3d::generate()
{
    QSettings settings;

    settings.insertSearchPath(QSettings::Unix,
			      QString(INSTALLDIR) + "/etc/");

    int collcount = settings.readNumEntry("/mq3d/Collections/count");
    if (! collcount) {
	qWarning("MQ3d: error, no collections defined, exiting.");
	err = TRUE;
	return;
    }

    QFile f(collpath);
    if (! f.open(IO_WriteOnly)) {
	qWarning("MQ3d: error, cannot open collections file, exiting.");
	err = TRUE;
	return;
    }

    QTextStream stream(&f);
    stream.setEncoding(QTextStream::UnicodeUTF8);
    stream << "<!DOCTYPE MQ3d-Collections><collections>" << endl;

    for (int i = 0; i < collcount; i++) {
	QString collection("Collection");
	collection += QString::number(i + 1);
	QString path = settings.readEntry(collection);
	addCollection(stream, collection, path);
    }
    stream << "</collections>" << endl;

    f.close();
}


void MQ3d::addCollection(QTextStream &stream, const QString &collection,
			 const QString &path)
{
    stream << "<collection>" << endl;
    stream << "    <name>" << collection << "</name>" << endl;

    doDir(stream, path, 0);

    stream << "</collection>" << endl;
}


static void doindent(QTextStream &stream, int indent)
{
    int i = indent;
    while (i--)
	stream << "    ";
}


void MQ3d::doDir(QTextStream &stream, const QString &path, int indent) {
    QDir dir(path);

    if (! dir.isReadable())
	return;

    QString s;

    if (indent) {
	doindent(stream, indent);
	stream << "<dir>" << endl;
	doindent(stream, indent + 1);

	s = dir.dirName();
	s = (s.replace(QRegExp("&"), "&amp;").
	     replace(QRegExp("<"), "&lt;").
	     replace(QRegExp(">"), "&gt;"));
	stream << "<name>" << s << "</name>" << endl;
    }

    const QFileInfoList *files = dir.entryInfoList();

    if (files) {
	QFileInfoListIterator it(*files);
	QFileInfo *fi;

	while ((fi = it.current()) != 0) {
	    ++it;

	    if (fi->fileName() == "." || fi->fileName() == "..")
		continue;

	    if (fi->isDir())
		doDir(stream, fi->absFilePath(), indent + 1);
	    else {
		doindent(stream, indent + 1);
		stream << "<file>" << endl;

		s = fi->fileName();
		s = (s.replace(QRegExp("&"), "&amp;").
		     replace(QRegExp("<"), "&lt;").
		     replace(QRegExp(">"), "&gt;"));
		doindent(stream, indent + 2);
		stream << "<name>" << s << "</name>" << endl;

		s = fi->absFilePath();
		s = (s.replace(QRegExp("&"), "&amp;").
		     replace(QRegExp("<"), "&lt;").
		     replace(QRegExp(">"), "&gt;"));
		doindent(stream, indent + 2);
		stream << "<path>" << s << "</path>" << endl;

		doindent(stream, indent + 1);
		stream << "</file>" << endl;
	    }
	}
    }

    if (indent) {
	doindent(stream, indent);
	stream << "</dir>" << endl;
    }
}
