---
page_type: sample
languages:
- python
products:
- azure
- cognitive-services
- azure-text-translator
name: azure.ai.translation.text samples for Python
description: Samples for the azure.ai.translation.text client library.
---

# Azure Text Translator client library for Python

Translator Service is a cloud-based neural machine translation service that is part of the Azure Cognitive Services family of REST APIs and can be used with any operating system. This client library offers the following features:

* Create Translation Client
* Get Supported Languages
* Translate
* Transliterate

See the [README][README] of the Text Translator client library for more information, including useful links and instructions.

# Create Client

Text Translation service is using two types of endpoints - Global and Custom. You can find more information in the [Translator reference][TranslatorReference].

## Global Endpoint

When using Text Translation service with global endpoint, the endpoint doesn't need to be provided when creating `TextTranslationClient`. When `endpoint` is
not provided, global `api.cognitive.microsofttranslator.com` is used. For [sovereign clouds][SovereignClouds], the endpoint is always required.

### Using the Subscription Key

When using `cognitive services key` and `region` with global endpoint, you can create `TextTranslationClient`:

<!-- SNIPPET: sample_text_translation_client.create_text_translation_client_with_credential -->

```python
credential = AzureKeyCredential(apikey)
text_translator = TextTranslationClient(credential=credential, region=region)
```

<!-- END SNIPPET -->

### Using the Cognitive Services Token

For the Cognitive Services Token authentication, there is currently no implementation provided in the SDK. You can implement the solution
according to the [Token Authentication documentation][TranslatorReference]. For a simple example implementation, you can refer: [StaticAccessTokenCredential][static_access_token_credential].
When the token is created, you can create the `TextTranslationClient`:

<!-- SNIPPET: sample_text_translation_client.create_text_translation_client_with_cognitive_services_token -->

```python
client = TextTranslationClient(credential=credential, audience="https://api.microsofttranslator.com/")
```

<!-- END SNIPPET -->

### Using the Entra Id Authentication

The Authentication with Microsoft Entra ID on global endpoint requires bearer token generated by Azure AD, Region of the translator resource
and Resource ID for your Translator resource instance. For prerequisites and more information refer to [Authentication with Microsoft Entra ID][TranslatorReference].

<!-- SNIPPET: sample_text_translation_client.create_text_translation_client_with_entra_id_token -->

```python
credential = DefaultAzureCredential()
client = TextTranslationClient(credential=credential, region=region, resource_id=resource_id)
```

<!-- END SNIPPET -->

## Custom Endpoint

When using some selected features (for example Virtual Network) you need to use custom endpoints. More information can be found in [Virtual Network support][TranslatorReference].

### Using the Subscription Key

For Custom endpoint and cognitive services key combination, you don't need to provide the region:

<!-- SNIPPET: sample_text_translation_client.create_text_translation_client_custom_with_credential -->

```python
credential = AzureKeyCredential(apikey)
text_translator = TextTranslationClient(credential=credential, endpoint=endpoint)
```

<!-- END SNIPPET -->

### Using the Cognitive Services Token

The Cognitive Services Token is not supported when using the Custom Endpoint.

### Using the Entra Id Authentication

The Authentication with Microsoft Entra ID on custom endpoint requires you to provide only bearer token generated by Azure AD:

<!-- SNIPPET: sample_text_translation_client.create_text_translation_client_custom_with_entra_id_token -->

```python
credential = DefaultAzureCredential()
client = TextTranslationClient(credential=credential, endpoint=endpoint)
```

<!-- END SNIPPET -->

## Azure AI Translator Container

When using `TextTranslationClient` with the [Azure AI Translator Container][Container] without any authentication. You will only need your endpoint:

<!-- SNIPPET: sample_text_translation_client.create_text_translation_client_with_endpoint -->

```python
text_translator = TextTranslationClient(endpoint=endpoint)
```

<!-- END SNIPPET -->

The values of the `endpoint` variable can be retrieved from environment variables, configuration settings, or any other secure approach that works for your application.

# Get Languages

This sample demonstrates how to get languages that are supported by other operations.

## Get Supported Languages for all other operations

This will return language metadata from all supported scopes.

<!-- SNIPPET: sample_text_translation_languages.get_text_translation_languages -->

```python
try:
    response = text_translator.get_supported_languages()

    print(
        f"Number of supported languages for translate operation: {len(response.translation) if response.translation is not None else 0}"
    )
    print(
        f"Number of supported languages for transliterate operation: {len(response.transliteration) if response.transliteration is not None else 0}"
    )
    print(
        f"Number of supported models for translation: {len(response.models) if response.models is not None else 0}"
    )

    if response.translation is not None:
        print("Translation Languages:")
        for key, value in response.translation.items():
            print(f"{key} -- name: {value.name} ({value.native_name})")

    if response.transliteration is not None:
        print("Transliteration Languages:")
        for key, value in response.transliteration.items():
            print(f"{key} -- name: {value.name}, supported script count: {len(value.scripts)}")

    if response.models is not None:
        print(f"Models: {', '.join(response.models)}")

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

### Get Supported Languages for a given scope

You can limit the scope of the response of the languages API by providing the optional parameter `scope`. A comma-separated list of names defining the group of languages to return. Allowed group names are: `translation`, `transliteration` and `dictionary`. If no scope is given, then all groups are returned, which is equivalent to passing `translation,transliteration,dictionary`.

<!-- SNIPPET: sample_text_translation_languages.get_text_translation_languages_scope -->

```python
try:
    scope = "translation"
    response = text_translator.get_supported_languages(scope=scope)

    print(
        f"Number of supported languages for translate operation: {len(response.translation) if response.translation is not None else 0}"
    )
    print(
        f"Number of supported languages for transliterate operation: {len(response.transliteration) if response.transliteration is not None else 0}"
    )
    print(
        f"Number of supported models for translation: {len(response.models) if response.models is not None else 0}"
    )

    if response.translation is not None:
        print("Translation Languages:")
        for key, value in response.translation.items():
            print(f"{key} -- name: {value.name} ({value.native_name})")

    if response.transliteration is not None:
        print("Transliteration Languages:")
        for key, value in response.transliteration.items():
            print(f"{key} -- name: {value.name}, supported script count: {len(value.scripts)}")

    if response.models is not None:
        print(f"Models: {', '.join(response.models)}")

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

### Get Languages in a given culture

You can select the language to use for user interface strings. Some of the fields in the response are names of languages or names of regions. Use this parameter to define the language in which these names are returned. The language is specified by providing a well-formed BCP 47 language tag. For instance, use the value `fr` to request names in French or use the value `zh-Hant` to request names in Chinese Traditional.
Names are provided in the English language when a target language is not specified or when localization is not available.

<!-- SNIPPET: sample_text_translation_languages.get_text_translation_languages_culture -->

```python
try:
    accept_language = "es"
    response = text_translator.get_supported_languages(accept_language=accept_language)

    print(
        f"Number of supported languages for translate operation: {len(response.translation) if response.translation is not None else 0}"
    )
    print(
        f"Number of supported languages for transliterate operation: {len(response.transliteration) if response.transliteration is not None else 0}"
    )
    print(
        f"Number of supported models for translation: {len(response.models) if response.models is not None else 0}"
    )

    if response.translation is not None:
        print("Translation Languages:")
        for key, value in response.translation.items():
            print(f"{key} -- name: {value.name} ({value.native_name})")

    if response.transliteration is not None:
        print("Transliteration Languages:")
        for key, value in response.transliteration.items():
            print(f"{key} -- name: {value.name}, supported script count: {len(value.scripts)}")

    if response.models is not None:
        print(f"Models: {', '.join(response.models)}")

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

# Translate

The following examples use the updated API that requires importing additional models:

```python
from azure.ai.translation.text.models import TranslateInputItem, TranslationTarget, TextType, ProfanityAction, ProfanityMarker
```

### Translate text

Translate text from known source language to target language.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation -->

```python
try:
    from_language = "en"
    to_language = ["cs"]
    input_text_elements = ["This is a test"]

    response = text_translator.translate(
        body=input_text_elements, to_language=to_language, from_language=from_language
    )
    translation = response[0] if response else None

    if translation:
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

### Translate with auto-detection

You can omit source language of the input text. In this case, API will try to auto-detect the language.

> Note that you must provide the source language rather than autodetection when using the dynamic dictionary feature.

> Note you can use `suggestedFrom` parameter that specifies a fallback language if the language of the input text can't be identified. Language autodetection is applied when the from parameter is omitted. If detection fails, the suggestedFrom language will be assumed.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_auto -->

```python
try:
    to_language = ["cs"]
    input_text_elements = ["This is a test"]

    response = text_translator.translate(body=input_text_elements, to_language=to_language)
    translation = response[0] if response else None

    if translation:
        detected_language = translation.detected_language
        if detected_language:
            print(
                f"Detected languages of the input text: {detected_language.language} with score: {detected_language.score}."
            )
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

### Translate with LLM

You can now translate using a large language model. You can also specify gender and tone using LLM translations.

> Using an LLM model requires you to have an Azure AI Foundry resource. For more information, see [Azure AI Translation Resources][Azure AI Translation Resources].

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_with_llm -->

```python
try:
    llm_model_name = "gpt-4o-mini"
    tone = "formal"
    gender = "female"
    to_language = "zh-Hans"
    input_text = "This is a test"
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[TranslationTarget(language=to_language, deployment_name=llm_model_name, tone=tone, gender=gender)],
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        detected_language = translation.detected_language
        if detected_language:
            print(
                f"Detected languages of the input text: {detected_language.language} with score: {detected_language.score}."
            )
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}' using LLM."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

### Translate with Transliteration

You can combine both Translation and Transliteration in one Translate call. Your source Text can be in non-standard Script of a language as well as you can ask for non-standard Script of a target language.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_with_transliteration -->

```python
try:
    from_script = "Latn"
    from_language = "ar"
    to_script = "Latn"
    to_language = "zh-Hans"
    input_text = "hudha akhtabar."
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[TranslationTarget(language=to_language, script=to_script)],
        language=from_language,
        script=from_script,
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        first_translation = translation.translations[0]
        if first_translation:
            print(f"Translation in transliteration script: '{first_translation.text}'.")

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

### Translate multiple input texts

You can translate multiple text elements with a various length. Each input element can be in different language (source language parameter needs to be omitted and language auto-detection is used). Refer to [Request limits for Translator](https://learn.microsoft.com/azure/cognitive-services/translator/request-limits) for current limits.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_multiple_inputs -->

```python
try:
    to_language = ["cs"]
    input_text_elements = [
        "This is a test.",
        "Esto es una prueba.",
        "Dies ist ein Test.",
    ]

    translations = text_translator.translate(body=input_text_elements, to_language=to_language)

    for translation in translations:
        print(
            f"Detected languages of the input text: {translation.detected_language.language if translation.detected_language else None} with score: {translation.detected_language.score if translation.detected_language else None}."
        )
        print(
            f"Text was translated to: '{translation.translations[0].language if translation.translations else None}' and the result is: '{translation.translations[0].text if translation.translations else None}'."
        )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

### Translate multiple target languages

You can provide multiple target languages which results to each input element be translated to all target languages.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_multiple_languages -->

```python
try:
    to_language = ["cs", "es", "de"]
    input_text_elements = ["This is a test"]

    response = text_translator.translate(body=input_text_elements, to_language=to_language)
    translation = response[0] if response else None

    if translation:
        detected_language = translation.detected_language
        if detected_language:
            print(
                f"Detected languages of the input text: {detected_language.language} with score: {detected_language.score}."
            )
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

### Translate different text types

You can select whether the translated text is plain text or HTML text. Any HTML needs to be a well-formed, complete element. Possible values are: plain (default) or html.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_type -->

```python
try:
    text_type = TextType.HTML
    to_language = "cs"
    input_text = "<html><body>This <b>is</b> a test.</body></html>"
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[TranslationTarget(language=to_language)],
        text_type=text_type,
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        detected_language = translation.detected_language
        if detected_language:
            print(
                f"Detected languages of the input text: {detected_language.language} with score: {detected_language.score}."
            )
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

### Don’t translate specific entity name in a text

It's sometimes useful to exclude specific content from translation. You can use the attribute class=notranslate to specify content that should remain in its original language. In the following example, the content inside the first div element won't be translated, while the content in the second div element will be translated.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_exclude -->

```python
try:
    text_type = TextType.HTML
    from_language = "en"
    to_language = "cs"
    input_text = '<div class="notranslate">This will not be translated.</div><div>This will be translated. </div>'
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[TranslationTarget(language=to_language)],
        language=from_language,
        text_type=text_type,
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

### Translate specific entity name in a text applying a dictionary

If you already know the translation you want to apply to a word or a phrase, you can supply it as markup within the request. The dynamic dictionary is safe only for compound nouns like proper names and product names.

> Note You must include the From parameter in your API translation request instead of using the autodetect feature.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_entity -->

```python
try:
    from_language = "en"
    to_language = "cs"
    input_text = 'The word <mstrans:dictionary translation="wordomatic">wordomatic</mstrans:dictionary> is a dictionary entry.'
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[TranslationTarget(language=to_language)],
        language=from_language,
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

### Profanity handling

[Profanity handling](https://learn.microsoft.com/azure/cognitive-services/translator/reference/v3-0-translate#handle-profanity). Normally the Translator service will retain profanity that is present in the source in the translation. The degree of profanity and the context that makes words profane differ between cultures, and as a result the degree of profanity in the target language may be amplified or reduced.

If you want to avoid getting profanity in the translation, regardless of the presence of profanity in the source text, you can use the profanity filtering option. The option allows you to choose whether you want to see profanity deleted, whether you want to mark profanities with appropriate tags (giving you the option to add your own post-processing), or you want no action taken. The accepted values of `ProfanityAction` are `DELETED`, `MARKED` and `NOACTION` (default).

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_profanity -->

```python
try:
    profanity_action = ProfanityAction.MARKED
    profanity_maker = ProfanityMarker.ASTERISK
    to_language = "cs"
    input_text = "This is ***."
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[
            TranslationTarget(
                language=to_language, profanity_action=profanity_action, profanity_marker=profanity_maker
            )
        ],
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        detected_language = translation.detected_language
        if detected_language:
            print(
                f"Detected languages of the input text: {detected_language.language} with score: {detected_language.score}."
            )
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

### Custom Translator

You can get translations from a customized system built with [Custom Translator](https://learn.microsoft.com/azure/cognitive-services/translator/customization). Add the Category ID from your Custom Translator [project details](https://learn.microsoft.com/azure/cognitive-services/translator/custom-translator/how-to-create-project#view-project-details) to this parameter to use your deployed customized system.

It is possible to set `allow_fallback` parameter. It specifies that the service is allowed to fall back to a general system when a custom system doesn't exist. Possible values are: `True` (default) or `False`.

`allow_fallback=False` specifies that the translation should only use systems trained for the category specified by the request. If a translation for language X to language Y requires chaining through a pivot language E, then all the systems in the chain (X → E and E → Y) will need to be custom and have the same category. If no system is found with the specific category, the request will return a 400 status code. `allow_fallback=True` specifies that the service is allowed to fall back to a general system when a custom system doesn't exist.

<!-- SNIPPET: sample_text_translation_translate.get_text_translation_custom -->

```python
try:
    category = "<<Category ID>>"
    to_language = "cs"
    input_text = "This is a test"
    input_text_element = TranslateInputItem(
        text=input_text,
        targets=[TranslationTarget(language=to_language, deployment_name=category)],
    )

    response = text_translator.translate(body=[input_text_element])
    translation = response[0] if response else None

    if translation:
        detected_language = translation.detected_language
        if detected_language:
            print(
                f"Detected languages of the input text: {detected_language.language} with score: {detected_language.score}."
            )
        for translated_text in translation.translations:
            print(
                f"Text was translated to: '{translated_text.language}' and the result is: '{translated_text.text}'."
            )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
```

<!-- END SNIPPET -->

# Transliterate

### Transliterate Text

Converts characters or letters of a source language to the corresponding characters or letters of a target language.

<!-- SNIPPET: sample_text_translation_transliterate.get_text_transliteration -->

```python
try:
    language = "zh-Hans"
    from_script = "Hans"
    to_script = "Latn"
    input_text_elements = ["这是个测试。"]

    response = text_translator.transliterate(
        body=input_text_elements,
        language=language,
        from_script=from_script,
        to_script=to_script,
    )
    transliteration = response[0] if response else None

    if transliteration:
        print(
            f"Input text was transliterated to '{transliteration.script}' script. Transliterated text: '{transliteration.text}'."
        )

except HttpResponseError as exception:
    if exception.error is not None:
        print(f"Error Code: {exception.error.code}")
        print(f"Message: {exception.error.message}")
    raise
```

<!-- END SNIPPET -->

* [Create Client][client_sample]
* [Translate][translate_sample]
* [Transliterate][transliterate_sample]

[README]: https://github.com/Azure/azure-sdk-for-python/tree/main/sdk/translation/azure-ai-translation-text/README.md
[client_sample]: https://github.com/Azure/azure-sdk-for-python/tree/main/sdk/translation/azure-ai-translation-text/samples/sample_text_translation_client.py
[languages_sample]: https://github.com/Azure/azure-sdk-for-python/tree/main/sdk/translation/azure-ai-translation-text/samples/sample_text_translation_languages.py
[translate_sample]: https://github.com/Azure/azure-sdk-for-python/tree/main/sdk/translation/azure-ai-translation-text/samples/sample_text_translation_translate.py
[transliterate_sample]: https://github.com/Azure/azure-sdk-for-python/tree/main/sdk/translation/azure-ai-translation-text/samples/sample_text_translation_transliterate.py
[static_access_token_credential]: https://github.com/Azure/azure-sdk-for-python/blob/main/sdk/translation/azure-ai-translation-text/tests/static_access_token_credential.py

[Container]: https://learn.microsoft.com/azure/ai-services/translator/containers/overview
[TranslatorReference]: https://learn.microsoft.com/azure/ai-services/translator/text-translation/preview/overview
[SovereignClouds]: https://learn.microsoft.com/azure/ai-services/translator/sovereign-clouds
[Azure AI Translation Resources]: https://learn.microsoft.com/azure/ai-services/translator/how-to/create-translator-resource?tabs=foundry