/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// StdLib includes
#include <tuple>


/////////////////////// Qt includes
#include <QDebug>
#include <QObject>
#include <QAction>
#include <QSettings>
#include <QMainWindow>

/////////////////////// pappsomspp includes
#include <pappsomspp/core/processing/detection/localsignaltonoiseestimator.h>
#include <pappsomspp/core/processing/detection/highrespeakpicker.h>

/////////////////////// libXpertMassCore includes
#include <MsXpS/libXpertMassCore/LowMassDeconvolver.hpp>

/////////////////////// Local includes


namespace Ui
{
class ApplicationPreferencesWnd;
class ApplicationPreferencesWndGeneralPageWidget;
class ApplicationPreferencesWndDiscoveriesPageWidget;
class DecimalPlacesPreferencesWidget;
class MaxThreadCountPreferencesWidget;
class PeakPickingPreferencesWidget;
class LowMassDeconvolverPreferencesWidget;
} // namespace Ui

namespace MsXpS
{
namespace MineXpert
{


class ApplicationPreferencesWnd: public QMainWindow
{
  Q_OBJECT

  public:
  enum Pages
  {
    GENERAL = 0x0,
    SHORTCUTS,
    DISCOVERIES,
    COUNT
  };

  ApplicationPreferencesWnd(const QString &application_name,
                            const QString &description,
                            QWidget *parent_p = nullptr);

  virtual ~ApplicationPreferencesWnd();

  void writeSettings() const;
  void readSettings();

  void setupWindow();

  // The Application preferences General page
  void setupGeneralPage();
  void setupMaxThreadCountWidget();
  void setupDecimalPlacesPreferencesWidget();
  void setupPeakPickingPreferencesWidget();
  void setupLowMassDeconvolverPreferencesWidget();

  // The Application preferences Discoveries page
  void setupDiscoveriesPage();

  // The Application preferences Shortcuts page
  void setupShortcutsPage();

  int insertSectionListItem(const QString &label,
                            const QIcon &icon,
                            QWidget *section_widget_p,
                            int index);

  std::tuple<pappso::LocalSignalToNoiseEstimator::Parameters,
             pappso::HighResPeakPicker::Parameters>
  getPeakPickingPreferences() const;

  bool getLowMassDeconvolverPreferences(
    libXpertMassCore::LowMassDeconvolver::Parameters &params) const;
  int populateLowMassDeconvolverPreferencesCombobox();
  bool populateLowMassDeconvolverPreferencesForHistoryItem(
    const QString &params_title);
  bool updateLowMassDeconvolverPreferencesCurrentHistoryItem();

  public slots:
  void apply();
  void addLowMassDeconvolverPreferencesToHistory();
  void removeLowMassDeconvolverPreferencesFromHistory();

  protected:
  Ui::ApplicationPreferencesWnd *mp_ui;
  Ui::ApplicationPreferencesWndGeneralPageWidget *mp_generalPage_ui;
  Ui::ApplicationPreferencesWndDiscoveriesPageWidget *mp_discoveriesPage_ui;
  Ui::DecimalPlacesPreferencesWidget *mp_decimalPlacesPreferencesWidget_ui;
  Ui::MaxThreadCountPreferencesWidget *mp_maxThreadCountPreferencesWidget_ui;
  Ui::PeakPickingPreferencesWidget *mp_peakPickingPreferencesWidget_ui;
  Ui::LowMassDeconvolverPreferencesWidget
    *mp_lowMassDeconvolverPreferencesWidget_ui;

  QMap<QString, libXpertMassCore::LowMassDeconvolver::Parameters>
    m_lowMassDeconvolverParameterPairs;

  QString m_applicationName;
  QString m_windowDescription;

  void applyGeneralPage();
  void writePeakPickingPreferences();
  void readPeakPickingPreferences();
  void writeLowMassDeconvolverPreferences();
  void readLowMassDeconvolverPreferences();
  void applyShortcutsPage();
  void applyDiscoveriesPage();
};


} // namespace MineXpert
} // namespace MsXpS
