/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.hive;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.api.LockComponent;
import org.apache.hadoop.hive.metastore.api.LockLevel;
import org.apache.hadoop.hive.metastore.api.LockRequest;
import org.apache.hadoop.hive.metastore.api.LockResponse;
import org.apache.hadoop.hive.metastore.api.LockState;
import org.apache.hadoop.hive.metastore.api.LockType;
import org.apache.hadoop.hive.metastore.api.ShowLocksRequest;
import org.apache.hadoop.hive.metastore.api.ShowLocksResponse;
import org.apache.hadoop.hive.metastore.api.ShowLocksResponseElement;
import org.apache.iceberg.ClientPool;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.hive.HiveHadoopUtil;
import org.apache.iceberg.hive.HiveLock;
import org.apache.iceberg.hive.HiveVersion;
import org.apache.iceberg.hive.LockException;
import org.apache.iceberg.relocated.com.google.common.base.MoreObjects;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.iceberg.util.Tasks;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MetastoreLock
implements HiveLock {
    private static final Logger LOG = LoggerFactory.getLogger(MetastoreLock.class);
    private static final String HIVE_ACQUIRE_LOCK_TIMEOUT_MS = "iceberg.hive.lock-timeout-ms";
    private static final String HIVE_LOCK_CHECK_MIN_WAIT_MS = "iceberg.hive.lock-check-min-wait-ms";
    private static final String HIVE_LOCK_CHECK_MAX_WAIT_MS = "iceberg.hive.lock-check-max-wait-ms";
    private static final String HIVE_LOCK_CREATION_TIMEOUT_MS = "iceberg.hive.lock-creation-timeout-ms";
    private static final String HIVE_LOCK_CREATION_MIN_WAIT_MS = "iceberg.hive.lock-creation-min-wait-ms";
    private static final String HIVE_LOCK_CREATION_MAX_WAIT_MS = "iceberg.hive.lock-creation-max-wait-ms";
    private static final String HIVE_LOCK_HEARTBEAT_INTERVAL_MS = "iceberg.hive.lock-heartbeat-interval-ms";
    private static final String HIVE_TABLE_LEVEL_LOCK_EVICT_MS = "iceberg.hive.table-level-lock-evict-ms";
    private static final long HIVE_ACQUIRE_LOCK_TIMEOUT_MS_DEFAULT = 180000L;
    private static final long HIVE_LOCK_CHECK_MIN_WAIT_MS_DEFAULT = 50L;
    private static final long HIVE_LOCK_CHECK_MAX_WAIT_MS_DEFAULT = 5000L;
    private static final long HIVE_LOCK_CREATION_TIMEOUT_MS_DEFAULT = 180000L;
    private static final long HIVE_LOCK_CREATION_MIN_WAIT_MS_DEFAULT = 50L;
    private static final long HIVE_LOCK_CREATION_MAX_WAIT_MS_DEFAULT = 5000L;
    private static final long HIVE_LOCK_HEARTBEAT_INTERVAL_MS_DEFAULT = 240000L;
    private static final long HIVE_TABLE_LEVEL_LOCK_EVICT_MS_DEFAULT = TimeUnit.MINUTES.toMillis(10L);
    private static volatile Cache<String, ReentrantLock> commitLockCache;
    private final ClientPool<IMetaStoreClient, TException> metaClients;
    private final String databaseName;
    private final String tableName;
    private final String fullName;
    private final long lockAcquireTimeout;
    private final long lockCheckMinWaitTime;
    private final long lockCheckMaxWaitTime;
    private final long lockCreationTimeout;
    private final long lockCreationMinWaitTime;
    private final long lockCreationMaxWaitTime;
    private final long lockHeartbeatIntervalTime;
    private final ScheduledExecutorService exitingScheduledExecutorService;
    private final String agentInfo;
    private Optional<Long> hmsLockId = Optional.empty();
    private ReentrantLock jvmLock = null;
    private Heartbeat heartbeat = null;

    MetastoreLock(Configuration conf, ClientPool<IMetaStoreClient, TException> metaClients, String catalogName, String databaseName, String tableName) {
        this.metaClients = metaClients;
        this.fullName = catalogName + "." + databaseName + "." + tableName;
        this.databaseName = databaseName;
        this.tableName = tableName;
        this.lockAcquireTimeout = conf.getLong(HIVE_ACQUIRE_LOCK_TIMEOUT_MS, 180000L);
        this.lockCheckMinWaitTime = conf.getLong(HIVE_LOCK_CHECK_MIN_WAIT_MS, 50L);
        this.lockCheckMaxWaitTime = conf.getLong(HIVE_LOCK_CHECK_MAX_WAIT_MS, 5000L);
        this.lockCreationTimeout = conf.getLong(HIVE_LOCK_CREATION_TIMEOUT_MS, 180000L);
        this.lockCreationMinWaitTime = conf.getLong(HIVE_LOCK_CREATION_MIN_WAIT_MS, 50L);
        this.lockCreationMaxWaitTime = conf.getLong(HIVE_LOCK_CREATION_MAX_WAIT_MS, 5000L);
        this.lockHeartbeatIntervalTime = conf.getLong(HIVE_LOCK_HEARTBEAT_INTERVAL_MS, 240000L);
        long tableLevelLockCacheEvictionTimeout = conf.getLong(HIVE_TABLE_LEVEL_LOCK_EVICT_MS, HIVE_TABLE_LEVEL_LOCK_EVICT_MS_DEFAULT);
        this.agentInfo = "Iceberg-" + String.valueOf(UUID.randomUUID());
        this.exitingScheduledExecutorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("iceberg-hive-lock-heartbeat-" + this.fullName + "-%d").build());
        MetastoreLock.initTableLevelLockCache(tableLevelLockCacheEvictionTimeout);
    }

    @Override
    public void lock() throws LockException {
        this.acquireJvmLock();
        this.hmsLockId = Optional.of(this.acquireLock());
        this.heartbeat = new Heartbeat(this.metaClients, this.hmsLockId.get(), this.lockHeartbeatIntervalTime);
        this.heartbeat.schedule(this.exitingScheduledExecutorService);
    }

    @Override
    public void ensureActive() throws LockException {
        if (this.heartbeat == null) {
            throw new LockException("Lock is not active", new Object[0]);
        }
        if (this.heartbeat.encounteredException != null) {
            throw new LockException(this.heartbeat.encounteredException, "Failed to heartbeat for hive lock. %s", this.heartbeat.encounteredException.getMessage());
        }
        if (!this.heartbeat.active()) {
            throw new LockException("Hive lock heartbeat thread not active", new Object[0]);
        }
    }

    @Override
    public void unlock() {
        if (this.heartbeat != null) {
            this.heartbeat.cancel();
            this.exitingScheduledExecutorService.shutdown();
        }
        try {
            this.unlock(this.hmsLockId);
        }
        finally {
            this.releaseJvmLock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long acquireLock() throws LockException {
        LockInfo lockInfo = this.createLock();
        long start = System.currentTimeMillis();
        long duration = 0L;
        boolean timeout = false;
        TException thriftError = null;
        try {
            if (lockInfo.lockState.equals((Object)LockState.WAITING)) {
                Tasks.foreach((Object[])new Long[]{lockInfo.lockId}).retry(2147483547).exponentialBackoff(this.lockCheckMinWaitTime, this.lockCheckMaxWaitTime, this.lockAcquireTimeout, 1.5).throwFailureWhenFinished().onlyRetryOn(WaitingForLockException.class).run(id -> {
                    try {
                        LockState newState;
                        LockResponse response = (LockResponse)this.metaClients.run(client -> client.checkLock(id.longValue()));
                        lockInfo.lockState = newState = response.getState();
                        if (newState.equals((Object)LockState.WAITING)) {
                            throw new WaitingForLockException(String.format("Waiting for lock on table %s.%s", this.databaseName, this.tableName));
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.interrupted();
                        LOG.warn("Interrupted while waiting for lock on table {}.{}", new Object[]{this.databaseName, this.tableName, e});
                    }
                }, TException.class);
            }
        }
        catch (WaitingForLockException e) {
            timeout = true;
            duration = System.currentTimeMillis() - start;
        }
        catch (TException e) {
            thriftError = e;
        }
        finally {
            if (!lockInfo.lockState.equals((Object)LockState.ACQUIRED)) {
                this.unlock(Optional.of(lockInfo.lockId));
            }
        }
        if (!lockInfo.lockState.equals((Object)LockState.ACQUIRED)) {
            if (timeout) {
                throw new LockException("Timed out after %s ms waiting for lock on %s.%s", duration, this.databaseName, this.tableName);
            }
            if (thriftError != null) {
                throw new LockException(thriftError, "Metastore operation failed for %s.%s", this.databaseName, this.tableName);
            }
            throw new LockException("Could not acquire the lock on %s.%s, lock request ended in state %s", this.databaseName, this.tableName, lockInfo.lockState);
        }
        return lockInfo.lockId;
    }

    private LockInfo createLock() throws LockException {
        String hostName;
        LockInfo lockInfo = new LockInfo();
        try {
            hostName = InetAddress.getLocalHost().getHostName();
        }
        catch (UnknownHostException uhe) {
            throw new LockException(uhe, "Error generating host name", new Object[0]);
        }
        LockComponent lockComponent = new LockComponent(LockType.EXCLUSIVE, LockLevel.TABLE, this.databaseName);
        lockComponent.setTablename(this.tableName);
        LockRequest lockRequest = new LockRequest((List)Lists.newArrayList((Object[])new LockComponent[]{lockComponent}), HiveHadoopUtil.currentUser(), hostName);
        if (HiveVersion.min(HiveVersion.HIVE_2)) {
            lockRequest.setAgentInfo(this.agentInfo);
        }
        AtomicBoolean interrupted = new AtomicBoolean(false);
        Tasks.foreach((Object[])new LockRequest[]{lockRequest}).retry(2147483547).exponentialBackoff(this.lockCreationMinWaitTime, this.lockCreationMaxWaitTime, this.lockCreationTimeout, 2.0).shouldRetryTest(e -> !interrupted.get() && e instanceof LockException && HiveVersion.min(HiveVersion.HIVE_2)).throwFailureWhenFinished().run(request -> {
            try {
                LockResponse lockResponse = (LockResponse)this.metaClients.run(client -> client.lock(request));
                lockInfo.lockId = lockResponse.getLockid();
                lockInfo.lockState = lockResponse.getState();
            }
            catch (TException te) {
                LOG.warn("Failed to create lock {}", request, (Object)te);
                try {
                    LockInfo lockFound;
                    if (HiveVersion.min(HiveVersion.HIVE_2) && (lockFound = this.findLock()) != null) {
                        lockInfo.lockId = lockFound.lockId;
                        lockInfo.lockState = lockFound.lockState;
                        LOG.info("Found lock {} by agentInfo {}", (Object)lockInfo, (Object)this.agentInfo);
                        return;
                    }
                    throw new LockException("Failed to find lock for table %s.%s", this.databaseName, this.tableName);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    interrupted.set(true);
                    LOG.warn("Interrupted while trying to find lock for table {}.{}", new Object[]{this.databaseName, this.tableName, e});
                    throw new LockException(e, "Interrupted while trying to find lock for table %s.%s", this.databaseName, this.tableName);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                interrupted.set(true);
                LOG.warn("Interrupted while creating lock on table {}.{}", new Object[]{this.databaseName, this.tableName, e});
                throw new LockException(e, "Interrupted while creating lock on table %s.%s", this.databaseName, this.tableName);
            }
        }, LockException.class);
        LOG.debug("Lock {} created for table {}.{}", new Object[]{lockInfo, this.databaseName, this.tableName});
        return lockInfo;
    }

    private LockInfo findLock() throws LockException, InterruptedException {
        ShowLocksResponse response;
        Preconditions.checkArgument((boolean)HiveVersion.min(HiveVersion.HIVE_2), (Object)"Minimally Hive 2 HMS client is needed to find the Lock using the showLocks API call");
        ShowLocksRequest showLocksRequest = new ShowLocksRequest();
        showLocksRequest.setDbname(this.databaseName);
        showLocksRequest.setTablename(this.tableName);
        try {
            response = (ShowLocksResponse)this.metaClients.run(client -> client.showLocks(showLocksRequest));
        }
        catch (TException e) {
            throw new LockException(e, "Failed to find lock for table %s.%s", this.databaseName, this.tableName);
        }
        for (ShowLocksResponseElement lock : response.getLocks()) {
            if (!lock.getAgentInfo().equals(this.agentInfo)) continue;
            return new LockInfo(lock.getLockid(), lock.getState());
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void unlock(Optional<Long> lockId) {
        Long id = null;
        try {
            if (!lockId.isPresent()) {
                if (!HiveVersion.min(HiveVersion.HIVE_2)) {
                    LOG.warn("Could not find lock with HMSClient {}", (Object)HiveVersion.current());
                    return;
                }
                LockInfo lockInfo = this.findLock();
                if (lockInfo == null) {
                    LOG.info("No lock found with {} agentInfo", (Object)this.agentInfo);
                    return;
                }
                id = lockInfo.lockId;
            } else {
                id = lockId.get();
            }
            this.doUnlock(id);
            return;
        }
        catch (InterruptedException ie) {
            if (id == null) {
                Thread.currentThread().interrupt();
                LOG.warn("Interrupted finding locks to unlock {}.{}", new Object[]{this.databaseName, this.tableName, ie});
                return;
            }
            try {
                Thread.interrupted();
                LOG.warn("Interrupted unlock we try one more time {}.{}", new Object[]{this.databaseName, this.tableName, ie});
                this.doUnlock(id);
                return;
            }
            catch (Exception e) {
                LOG.warn("Failed to unlock even on 2nd attempt {}.{}", new Object[]{this.databaseName, this.tableName, e});
                return;
            }
            finally {
                Thread.currentThread().interrupt();
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to unlock {}.{}", new Object[]{this.databaseName, this.tableName, e});
        }
    }

    private void doUnlock(long lockId) throws TException, InterruptedException {
        this.metaClients.run(client -> {
            client.unlock(lockId);
            return null;
        });
    }

    private void acquireJvmLock() {
        if (this.jvmLock != null) {
            throw new IllegalStateException(String.format("Cannot call acquireLock twice for %s", this.fullName));
        }
        this.jvmLock = (ReentrantLock)commitLockCache.get((Object)this.fullName, t -> new ReentrantLock(true));
        this.jvmLock.lock();
    }

    private void releaseJvmLock() {
        if (this.jvmLock != null) {
            this.jvmLock.unlock();
            this.jvmLock = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void initTableLevelLockCache(long evictionTimeout) {
        if (commitLockCache != null) return;
        Class<MetastoreLock> clazz = MetastoreLock.class;
        synchronized (MetastoreLock.class) {
            if (commitLockCache != null) return;
            commitLockCache = Caffeine.newBuilder().expireAfterAccess(evictionTimeout, TimeUnit.MILLISECONDS).build();
            // ** MonitorExit[var2_1] (shouldn't be in output)
            return;
        }
    }

    private static class WaitingForLockException
    extends RuntimeException {
        WaitingForLockException(String message) {
            super(message);
        }
    }

    private static class LockInfo {
        private long lockId;
        private LockState lockState;

        private LockInfo() {
            this.lockId = -1L;
            this.lockState = null;
        }

        private LockInfo(long lockId, LockState lockState) {
            this.lockId = lockId;
            this.lockState = lockState;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("lockId", this.lockId).add("lockState", (Object)this.lockState).toString();
        }
    }

    private static class Heartbeat
    implements Runnable {
        private final ClientPool<IMetaStoreClient, TException> hmsClients;
        private final long lockId;
        private final long intervalMs;
        private ScheduledFuture<?> future;
        private volatile Exception encounteredException = null;

        Heartbeat(ClientPool<IMetaStoreClient, TException> hmsClients, long lockId, long intervalMs) {
            this.hmsClients = hmsClients;
            this.lockId = lockId;
            this.intervalMs = intervalMs;
            this.future = null;
        }

        @Override
        public void run() {
            try {
                this.hmsClients.run(client -> {
                    client.heartbeat(0L, this.lockId);
                    return null;
                });
            }
            catch (InterruptedException | TException e) {
                this.encounteredException = e;
                throw new CommitFailedException(e, "Failed to heartbeat for lock: %d", new Object[]{this.lockId});
            }
        }

        public void schedule(ScheduledExecutorService scheduler) {
            this.future = scheduler.scheduleAtFixedRate(this, this.intervalMs / 2L, this.intervalMs, TimeUnit.MILLISECONDS);
        }

        boolean active() {
            return this.future != null && !this.future.isCancelled();
        }

        public void cancel() {
            if (this.future != null) {
                this.future.cancel(false);
            }
        }
    }
}

