# -*- coding: utf-8 -*-

# settings.py
# This file is part of qarte-5
#    
# Author: Vincent Vande Vyvre <vincent.vandevyvre@oqapy.eu>
# Copyright: 2012-2022 Vincent Vande Vyvre
# Licence: GPL3
# Home page: https://launchpad.net/qarte
#
# Settings dialog box

import sys
import os

from PyQt5.QtCore import Qt, QSize
from PyQt5.QtGui import QPixmap
from PyQt5.QtWidgets import (QDialog, QGridLayout, QVBoxLayout, QHBoxLayout,
                             QLabel, QComboBox, QCheckBox, QSpacerItem,
                             QSizePolicy, QApplication, QLineEdit, QPushButton,
                             QFrame, QFileDialog, )

from data import VERSION_STR

class Settings(QDialog):
    def __init__(self, core):
        super().__init__()
        self.core = core
        self.setWindowTitle("Settings - Qarte-%s" % VERSION_STR)
        self.resize(600, 400)
        self.setSizeGripEnabled(True)
        self.gridLayout = QGridLayout(self)
        self.gridLayout.setContentsMargins(6, 6, 6, 6)
        vl = QVBoxLayout()
        hl = QHBoxLayout()
        self.start_lbl = QLabel(self)
        self.start_lbl.setText(_("Load page:"))
        self.lang_cmb = QComboBox(self)
        self.lang_cmb.addItems(["Français", "Deutsch"])
        sp = QSpacerItem(40, 13, QSizePolicy.Expanding, QSizePolicy.Minimum)
        hl.addWidget(self.start_lbl)
        hl.addWidget(self.lang_cmb)
        hl.addItem(sp)
        vl.addLayout(hl)
        self.gridLayout.addLayout(vl, 0, 0, 1, 1)
        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        vl.addWidget(line)
        hl3 = QHBoxLayout()
        self.logo_lbl = QLabel(self)
        self.logo_lbl.setPixmap(QPixmap("medias/logo+7_112x48.png"))
        hl3.addWidget(self.logo_lbl)
        vl.addLayout(hl3)
        hl4 = QHBoxLayout()
        self.folder_lbl = QLabel(self)
        self.folder_lbl.setText(_("Videos folder:"))
        self.folder_led = QLineEdit(self)
        self.browse1_btn = QPushButton(self)
        self.browse1_btn.setText(_("Browse"))
        hl4.addWidget(self.folder_lbl)
        hl4.addWidget(self.folder_led)
        hl4.addWidget(self.browse1_btn)
        vl.addLayout(hl4)
        hl5 = QHBoxLayout()
        self.warning_lbl = QLabel(self)
        self.warning_lbl.setAlignment(Qt.AlignCenter)
        self.warning_lbl.setText(_("Warning, this folder is read only!"))
        self.warning_lbl.setStyleSheet("QFrame {color: #FF0000;}")
        self.warning_lbl.hide()
        hl5.addWidget(self.warning_lbl)
        vl.addLayout(hl5)
        hl15 = QHBoxLayout()
        self.add_date_chb = QCheckBox(self)
        self.add_date_chb.setText(_("Add the date to the file name"))
        hl15.addWidget(self.add_date_chb)
        vl.addLayout(hl15)
        hl6 = QHBoxLayout()
        self.summary_chb = QCheckBox(self)
        self.summary_chb.setText(_("Record the summaries"))
        self.summary_cmb = QComboBox(self)
        self.summary_cmb.addItems([_("Use one file by movie"),
                                    _("Use one unique file")])
        self.summary_cmb.setEnabled(False)
        sp2 = QSpacerItem(40, 13, QSizePolicy.Expanding, QSizePolicy.Minimum)
        hl6.addWidget(self.summary_chb)
        hl6.addWidget(self.summary_cmb)
        hl6.addItem(sp2)
        vl.addLayout(hl6)
        hl7 = QHBoxLayout()
        self.copy_thumbs_chb = QCheckBox(self)
        self.copy_thumbs_chb.setText(_("Copy a thumbnail for each video"))
        sp3 = QSpacerItem(40, 13, QSizePolicy.Expanding, QSizePolicy.Minimum)
        hl7.addWidget(self.copy_thumbs_chb)
        hl7.addItem(sp3)
        vl.addLayout(hl7)
        hl8 = QHBoxLayout()
        line1 = QFrame(self)
        line1.setFrameShape(QFrame.HLine)
        line1.setFrameShadow(QFrame.Sunken)
        vl.addWidget(line1)
        hl9 = QHBoxLayout()
        self.logo1_lbl = QLabel(self)
        self.logo1_lbl.setPixmap(QPixmap("medias/logoLW_213x36.png"))
        hl9.addWidget(self.logo1_lbl)
        vl.addLayout(hl9)
        hl10 = QHBoxLayout()
        self.folder1_lbl = QLabel(self)
        self.folder1_lbl.setText(_("Music folder:"))
        self.folder1_led = QLineEdit(self)
        self.browse2_btn = QPushButton(self)
        self.browse2_btn.setText(_("Browse"))
        hl10.addWidget(self.folder1_lbl)
        hl10.addWidget(self.folder1_led)
        hl10.addWidget(self.browse2_btn)
        vl.addLayout(hl10)
        hl11 = QHBoxLayout()
        self.warning1_lbl = QLabel(self)
        self.warning1_lbl.setAlignment(Qt.AlignCenter)
        self.warning1_lbl.setText(_("Warning, this folder is read only!"))
        self.warning1_lbl.setStyleSheet("QFrame {color: #FF0000;}")
        self.warning1_lbl.hide()
        hl11.addWidget(self.warning1_lbl)
        vl.addLayout(hl11)
        hl12 = QHBoxLayout()
        self.summary1_chb = QCheckBox(self)
        self.summary1_chb.setText(_("Record the summaries"))
        self.summary1_cmb = QComboBox(self)
        self.summary1_cmb.addItems([_("Use one file by movie"),
                                    _("Use one unique file")])
        self.summary1_cmb.setEnabled(False)
        sp4 = QSpacerItem(40, 13, QSizePolicy.Expanding, QSizePolicy.Minimum)
        hl12.addWidget(self.summary1_chb)
        hl12.addWidget(self.summary1_cmb)
        hl12.addItem(sp4)
        vl.addLayout(hl12)
        hl13 = QHBoxLayout()
        self.copy_thumbs1_chb = QCheckBox(self)
        self.copy_thumbs1_chb.setText(_("Copy a thumbnail for each video"))
        sp5 = QSpacerItem(40, 13, QSizePolicy.Expanding, QSizePolicy.Minimum)
        hl13.addWidget(self.copy_thumbs1_chb)
        hl13.addItem(sp5)
        vl.addLayout(hl13)
        hl14 = QHBoxLayout()
        sp5 = QSpacerItem(40, 13, QSizePolicy.Expanding, QSizePolicy.Minimum)
        hl14.addItem(sp5)
        self.close_btn = QPushButton(self)
        self.close_btn.setText(_("Close"))
        hl14.addWidget(self.close_btn)
        vl.addLayout(hl14)
        self.summary_chb.stateChanged.connect(self.summary_cmb.setEnabled)
        self.summary1_chb.stateChanged.connect(self.summary1_cmb.setEnabled)

    def configure(self, cfg):
        self.cfg = cfg
        self.lang_cmb.setCurrentIndex(['fr', 'de'].index(cfg.get('lang')))
        self.folder_led.setText(cfg.get('videos_folder'))
        self.add_date_chb.setChecked(cfg.get('add_date'))
        self.summary_chb.setChecked(cfg.get('pitch_plus'))
        self.summary_cmb.setCurrentIndex(cfg.get('pitch_plus_unique'))
        self.copy_thumbs_chb.setChecked(cfg.get('copythumb_plus'))
        self.folder1_led.setText(cfg.get('music_folder'))
        self.summary1_chb.setChecked(cfg.get('pitch_live'))
        self.summary1_cmb.setCurrentIndex(cfg.get('pitch_live_unique'))
        self.copy_thumbs1_chb.setChecked(cfg.get('copythumb_live'))
        if cfg.get('pitch_plus'):
            self.summary_chb.setEnabled(True)
        if cfg.get('pitch_live'):
            self.summary1_chb.setEnabled(True)

        self.folder_led.editingFinished.connect(self.on_videos_fld_changed)
        self.folder1_led.editingFinished.connect(self.on_music_fld_changed)
        self.browse1_btn.clicked.connect(self.get_videos_path)
        self.browse2_btn.clicked.connect(self.get_music_path)
        self.close_btn.clicked.connect(self.closeEvent)

    def on_videos_fld_changed(self):
        path = self.folder_led.text()
        if os.path.isdir(path):
            if not os.access(path, os.W_OK):
                # Read-only
                self.warning_lbl.show()

            else:
                self.warning_lbl.hide()

    def on_music_fld_changed(self):
        path = self.folder1_led.text()
        if os.path.isdir(path):
            if not os.access(path, os.W_OK):
                # Read-only
                self.warning1_lbl.show()

            else:
                self.warning1_lbl.hide()

    def get_videos_path(self):
        fld = self.get_folder()
        if fld is not None:
            self.folder_led.setText(fld)

    def get_music_path(self):
        fld = self.get_folder()
        if fld is not None:
            self.folder1_led.setText(fld)

    def get_folder(self):
        f = os.path.expanduser('~')
        d = QFileDialog.getExistingDirectory(None, 
                            "Choose destination folder", f, 
                            QFileDialog.DontResolveSymlinks
                            | QFileDialog.ShowDirsOnly)
        if not d:
            return

        if not os.path.isdir(d):
            if not self.create_folder(d):
                return

        return d

    def create_folder(self, path):
        try:
            os.makedirs(path)
        except Exception as why:
            self.core.show_warning(10,(path, why))
            return False

        return True

    def closeEvent(self, event=None):
        user = os.path.expanduser("~")
        self.cfg.update('lang', ['fr', 'de'][self.lang_cmb.currentIndex()])
        # Don't change old settings if empty
        if self.folder_led.text():
            if not self.warning_lbl.isVisible():
                self.cfg.update('videos_folder', self.folder_led.text())

        if not self.cfg.get('videos_folder'):
            self.cfg.update('videos_folder', user)

        self.cfg.update('add_date', self.add_date_chb.isChecked())
        self.cfg.update('pitch_plus', self.summary_chb.isChecked())
        self.cfg.update('pitch_plus_unique', self.summary_cmb.currentIndex())
        self.cfg.update('copythumb_plus', self.copy_thumbs_chb.isChecked())
        if self.folder1_led.text():
            if not self.warning1_lbl.isVisible():
                self.cfg.update('music_folder', self.folder1_led.text())

        if not self.cfg.get('music_folder'):
            self.cfg.update('music_folder', user)

        self.cfg.update('pitch_live', self.summary1_chb.isChecked())
        self.cfg.update('pitch_live_unique', self.summary1_cmb.currentIndex())
        self.cfg.update('copythumb_live', self.copy_thumbs1_chb.isChecked())
        self.accept()


if __name__ == "__main__":
    app = QApplication([])
    sett = Settings(None)
    sett.show()
    sys.exit(app.exec_())


