// license:BSD-3-Clause
// copyright-holders:R. Belmont
/*********************************************************************

    a2hsscsi.cpp

    Implementation of the Apple II High Speed SCSI Card

    This uses an ASIC called "Sandwich II"; the card itself is
    sometimes known as "Cocoon".

    Notes:
    C0n0-C0n7 = NCR5380 registers in normal order
    C0n8 = DMA address low
    C0n9 = DMA address high
    C0nA = DMA count low
    C0nB = DMA count high
    C0nC = DMA control
    C0nD = Enable DMA / reset 5380
    C0nE = Priority (read bits 5-7) / Fire watchdog (write bit 7) / RAM bank (write bits 0-3)
    C0nF = DMA speed (bit 7 = 0 for fast, 1 for slow) / ROM bank (write bits 0-4)

    DMA control register (C0nC):
    0x01 = pseudo-DMA enable
    0x02 = DMA enable
    0x04 = test mode
    0x08 = disable stop-DMA-on-IRQ
    0x10 = DMA direction (read only)
    0x20 = 5380 IRQ enable
    0x40 = system DMA status (read only)
    0x80 = DMA stopped due to IRQ

    Enable DMA / reset 5380 register (C0nD):
    0x01 = Resume DMA after rollover or IRQ
    0x02 = Reset the 5380
    0x40 = Clear test mode
    0x80 = Set test mode

*********************************************************************/

#include "emu.h"
#include "a2hsscsi.h"
#include "bus/nscsi/devices.h"

/***************************************************************************
    PARAMETERS
***************************************************************************/

//**************************************************************************
//  GLOBAL VARIABLES
//**************************************************************************

DEFINE_DEVICE_TYPE(A2BUS_HSSCSI, a2bus_hsscsi_device, "a2hsscsi", "Apple II High-Speed SCSI Card")

#define SCSI_ROM_REGION  "scsi_rom"
#define SCSI_BUS_TAG     "scsibus"
#define SCSI_5380_TAG    "scsibus:7:ncr5380"

ROM_START( hsscsi )
	ROM_REGION(0x8000, SCSI_ROM_REGION, 0)
	ROM_LOAD( "341-0803.bin", 0x0000, 0x8000, CRC(2c15618b) SHA1(7d32227299933bfc1b7f8bc2062906fdfe530674) )
ROM_END

/***************************************************************************
    FUNCTION PROTOTYPES
***************************************************************************/

//-------------------------------------------------
//  device_add_mconfig - add device configuration
//-------------------------------------------------

void a2bus_hsscsi_device::device_add_mconfig(machine_config &config)
{
	NSCSI_BUS(config, m_scsibus);
	NSCSI_CONNECTOR(config, "scsibus:0", default_scsi_devices, nullptr, false);
	NSCSI_CONNECTOR(config, "scsibus:1", default_scsi_devices, nullptr, false);
	NSCSI_CONNECTOR(config, "scsibus:2", default_scsi_devices, nullptr, false);
	NSCSI_CONNECTOR(config, "scsibus:3", default_scsi_devices, nullptr, false);
	NSCSI_CONNECTOR(config, "scsibus:4", default_scsi_devices, nullptr, false);
	NSCSI_CONNECTOR(config, "scsibus:5", default_scsi_devices, nullptr, false);
	NSCSI_CONNECTOR(config, "scsibus:6", default_scsi_devices, "harddisk", false);
	NSCSI_CONNECTOR(config, "scsibus:7").option_set("ncr5380", NCR53C80).machine_config([this](device_t *device) {
		downcast<ncr53c80_device &>(*device).drq_handler().set(*this, FUNC(a2bus_hsscsi_device::drq_w));
	});
}

//-------------------------------------------------
//  rom_region - device-specific ROM region
//-------------------------------------------------

const tiny_rom_entry *a2bus_hsscsi_device::device_rom_region() const
{
	return ROM_NAME( hsscsi );
}

//**************************************************************************
//  LIVE DEVICE
//**************************************************************************

a2bus_hsscsi_device::a2bus_hsscsi_device(const machine_config &mconfig, device_type type, const char *tag, device_t *owner, uint32_t clock) :
	device_t(mconfig, type, tag, owner, clock),
	device_a2bus_card_interface(mconfig, *this),
	m_ncr5380(*this, SCSI_5380_TAG),
	m_scsibus(*this, SCSI_BUS_TAG), m_rom(*this, SCSI_ROM_REGION), m_rambank(0), m_rombank(0), m_drq(0), m_bank(0), m_c0ne(0), m_c0nf(0),
	m_dma_addr(0), m_dma_size(0)
{
}

a2bus_hsscsi_device::a2bus_hsscsi_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock) :
	a2bus_hsscsi_device(mconfig, A2BUS_HSSCSI, tag, owner, clock)
{
}

//-------------------------------------------------
//  device_start - device-specific startup
//-------------------------------------------------

void a2bus_hsscsi_device::device_start()
{
	memset(m_ram, 0, 8192);

	save_item(NAME(m_ram));
	save_item(NAME(m_rambank));
	save_item(NAME(m_rombank));
	save_item(NAME(m_bank));
	save_item(NAME(m_drq));
	save_item(NAME(m_dma_addr));
	save_item(NAME(m_dma_size));
}

void a2bus_hsscsi_device::device_reset()
{
	reset_from_bus();
}

void a2bus_hsscsi_device::reset_from_bus()
{
	m_rambank = 0;
	m_rombank = 0;
	m_dma_addr = m_dma_size = 0;
	m_c0ne = m_c0nf = 0;

	m_ncr5380->reset();
}


/*-------------------------------------------------
    read_c0nx - called for reads from this card's c0nx space
-------------------------------------------------*/

uint8_t a2bus_hsscsi_device::read_c0nx(uint8_t offset)
{
	switch (offset)
	{
		case 0:
		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		case 7:
//          logerror("Read 5380 @ %x\n", offset);
			return m_ncr5380->read(offset);

		case 6:
			if (m_dma_control & 1)  // pseudo-DMA?
			{
				return m_ncr5380->dma_r();
			}
			else
			{
				//printf("Read 5380 PIO data (%s)\n", machine().describe_context().c_str());
				return m_ncr5380->read(offset);
			}

		case 8: // DMA address low
			return m_dma_addr & 0xff;

		case 9: // DMA address high
			return (m_dma_addr >> 8) & 0xff;

		case 0xa:   // DMA size low
			return m_dma_size & 0xff;

		case 0xb:   // DMA size high
			return (m_dma_size >> 8) & 0xff;

		case 0xc:   // DMA control
			return m_dma_control;

		case 0xe:   // code at cf32 wants to RMW this without killing the ROM bank
			return m_c0ne;

		case 0xf:
			return m_c0nf;

		default:
			logerror("Read c0n%x (%s)\n", offset, machine().describe_context().c_str());
			break;
	}

	return 0xff;
}


/*-------------------------------------------------
    write_c0nx - called for writes to this card's c0nx space
-------------------------------------------------*/

void a2bus_hsscsi_device::write_c0nx(uint8_t offset, uint8_t data)
{
	switch (offset)
	{
		case 0: // data out register; in PDMA mode, it's assumed this goes to DMA as reads do
			if (m_dma_control & 1)
			{
				m_ncr5380->dma_w(data);
			}
			else
			{
				m_ncr5380->write(offset, data);
			}
			break;

		case 1:
		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		case 7:
//          logerror("%02x to 5380 reg %x\n", data, offset);
			m_ncr5380->write(offset, data);
			break;

		case 8: // DMA address low
			//printf("%02x to DMA adr low\n", data);
			m_dma_addr &= 0xff00;
			m_dma_addr |= data;
			break;

		case 9: // DMA address high
			//printf("%02x to DMA adr high\n", data);
			m_dma_addr &= 0x00ff;
			m_dma_addr |= (data << 8);
			break;

		case 0xa: // DMA count low
			//printf("%02x to DMA count low\n", data);
			m_dma_size &= 0xff00;
			m_dma_size |= data;
			break;

		case 0xb: // DMA count high
			//printf("%02x to DMA count high\n", data);
			m_dma_size &= 0x00ff;
			m_dma_size |= (data << 8);
			break;

		case 0xc:   // DMA control
			//printf("%02x to DMA control\n", data);
			m_dma_control &= ~0x2b; // clear the read/write bits
			m_dma_control |= (data & 0x2b);
			break;

		case 0xd:   // DMA enable / reset
			//printf("%02x to DMA enable/reset\n", data);
			if (data & 0x2)
			{
	//          logerror("Resetting SCSI: %02x at %s\n", data, machine().describe_context());
				m_ncr5380->reset();
			}
			break;

		case 0xe:
			m_c0ne = data;
			m_rombank = (data & 0x1f) * 0x400;
			//logerror("c0ne to %x (ROM bank %x)\n", data & 0x1f, m_rombank);
			break;

		case 0xf:
			m_c0nf = data;
			m_rambank = (data & 0x7) * 0x400;
			//logerror("c0nf to %x (RAM bank %x)\n", data & 0x7, m_rambank);
			break;

		default:
			logerror("Write %02x to c0n%x (%s)\n", data, offset, machine().describe_context().c_str());
			break;
	}
}

/*-------------------------------------------------
    read_cnxx - called for reads from this card's cnxx space
-------------------------------------------------*/

uint8_t a2bus_hsscsi_device::read_cnxx(uint8_t offset)
{
	// one slot image at the start of the ROM, it appears
	return m_rom[offset];
}

void a2bus_hsscsi_device::write_cnxx(uint8_t offset, uint8_t data)
{
//  logerror("Write %02x to cn%02x (PC=%x)\n", data, offset, machine().describe_context());
}

/*-------------------------------------------------
    read_c800 - called for reads from this card's c800 space
-------------------------------------------------*/

uint8_t a2bus_hsscsi_device::read_c800(uint16_t offset)
{
	// bankswitched RAM at c800-cbff
	// bankswitched ROM at cc00-cfff
	if (offset < 0x400)
	{
//      logerror("Read RAM at %x = %02x\n", offset+m_rambank, m_ram[offset + m_rambank]);
		return m_ram[offset + m_rambank];
	}
	else
	{
		return m_rom[(offset-0x400) + m_rombank];
	}
}

/*-------------------------------------------------
    write_c800 - called for writes to this card's c800 space
-------------------------------------------------*/
void a2bus_hsscsi_device::write_c800(uint16_t offset, uint8_t data)
{
	if (offset < 0x400)
	{
//      logerror("%02x to RAM at %x\n", data, offset+m_rambank);
		m_ram[offset + m_rambank] = data;
	}
}

void a2bus_hsscsi_device::drq_w(int state)
{
	m_drq = (state ? 0x80 : 0x00);
//  printf("DRQ to %d\n", state);
}
