// SPDX-FileCopyrightText: Copyright 2022 yuzu Emulator Project
// SPDX-License-Identifier: GPL-2.0-or-later

#include "audio_core/renderer/command/resample/resample.h"

namespace AudioCore::Renderer {

static void ResampleLowQuality(std::span<s32> output, std::span<const s16> input,
                               const Common::FixedPoint<49, 15>& sample_rate_ratio,
                               Common::FixedPoint<49, 15>& fraction, const u32 samples_to_write) {
    if (sample_rate_ratio == 1.0f) {
        for (u32 i = 0; i < samples_to_write; i++) {
            output[i] = input[i];
        }
    } else {
        u32 read_index{0};
        for (u32 i = 0; i < samples_to_write; i++) {
            output[i] = input[read_index + (fraction >= 0.5f)];
            fraction += sample_rate_ratio;
            read_index += static_cast<u32>(fraction.to_int_floor());
            fraction.clear_int();
        }
    }
}

static void ResampleNormalQuality(std::span<s32> output, std::span<const s16> input,
                                  const Common::FixedPoint<49, 15>& sample_rate_ratio,
                                  Common::FixedPoint<49, 15>& fraction,
                                  const u32 samples_to_write) {
    static constexpr std::array<f32, 512> lut0 = {
        0.20141602f, 0.59283447f, 0.20513916f, 0.00009155f, 0.19772339f, 0.59277344f, 0.20889282f,
        0.00027466f, 0.19406128f, 0.59262085f, 0.21264648f, 0.00045776f, 0.19039917f, 0.59240723f,
        0.21646118f, 0.00067139f, 0.18679810f, 0.59213257f, 0.22030640f, 0.00085449f, 0.18322754f,
        0.59176636f, 0.22415161f, 0.00103760f, 0.17968750f, 0.59133911f, 0.22802734f, 0.00125122f,
        0.17617798f, 0.59085083f, 0.23193359f, 0.00146484f, 0.17269897f, 0.59027100f, 0.23583984f,
        0.00167847f, 0.16925049f, 0.58963013f, 0.23977661f, 0.00189209f, 0.16583252f, 0.58892822f,
        0.24374390f, 0.00210571f, 0.16244507f, 0.58816528f, 0.24774170f, 0.00234985f, 0.15908813f,
        0.58731079f, 0.25173950f, 0.00256348f, 0.15576172f, 0.58639526f, 0.25576782f, 0.00280762f,
        0.15249634f, 0.58541870f, 0.25979614f, 0.00308228f, 0.14923096f, 0.58435059f, 0.26385498f,
        0.00332642f, 0.14602661f, 0.58325195f, 0.26794434f, 0.00360107f, 0.14285278f, 0.58206177f,
        0.27203369f, 0.00387573f, 0.13973999f, 0.58078003f, 0.27612305f, 0.00418091f, 0.13662720f,
        0.57946777f, 0.28024292f, 0.00448608f, 0.13357544f, 0.57806396f, 0.28436279f, 0.00479126f,
        0.13052368f, 0.57662964f, 0.28851318f, 0.00512695f, 0.12753296f, 0.57510376f, 0.29266357f,
        0.00546265f, 0.12460327f, 0.57351685f, 0.29681396f, 0.00579834f, 0.12167358f, 0.57183838f,
        0.30099487f, 0.00616455f, 0.11880493f, 0.57012939f, 0.30517578f, 0.00656128f, 0.11596680f,
        0.56835938f, 0.30935669f, 0.00695801f, 0.11318970f, 0.56649780f, 0.31353760f, 0.00735474f,
        0.11041260f, 0.56457520f, 0.31771851f, 0.00778198f, 0.10769653f, 0.56262207f, 0.32192993f,
        0.00823975f, 0.10501099f, 0.56057739f, 0.32614136f, 0.00869751f, 0.10238647f, 0.55847168f,
        0.33032227f, 0.00915527f, 0.09976196f, 0.55633545f, 0.33453369f, 0.00967407f, 0.09722900f,
        0.55410767f, 0.33874512f, 0.01019287f, 0.09469604f, 0.55181885f, 0.34295654f, 0.01071167f,
        0.09222412f, 0.54949951f, 0.34713745f, 0.01126099f, 0.08978271f, 0.54708862f, 0.35134888f,
        0.01184082f, 0.08737183f, 0.54464722f, 0.35552979f, 0.01245117f, 0.08499146f, 0.54214478f,
        0.35974121f, 0.01306152f, 0.08267212f, 0.53958130f, 0.36392212f, 0.01370239f, 0.08041382f,
        0.53695679f, 0.36810303f, 0.01437378f, 0.07815552f, 0.53427124f, 0.37225342f, 0.01507568f,
        0.07595825f, 0.53155518f, 0.37640381f, 0.01577759f, 0.07379150f, 0.52877808f, 0.38055420f,
        0.01651001f, 0.07165527f, 0.52593994f, 0.38470459f, 0.01727295f, 0.06958008f, 0.52307129f,
        0.38882446f, 0.01806641f, 0.06753540f, 0.52014160f, 0.39294434f, 0.01889038f, 0.06552124f,
        0.51715088f, 0.39703369f, 0.01974487f, 0.06356812f, 0.51409912f, 0.40112305f, 0.02059937f,
        0.06164551f, 0.51101685f, 0.40518188f, 0.02148438f, 0.05975342f, 0.50790405f, 0.40921021f,
        0.02243042f, 0.05789185f, 0.50473022f, 0.41323853f, 0.02337646f, 0.05609131f, 0.50152588f,
        0.41726685f, 0.02435303f, 0.05432129f, 0.49826050f, 0.42123413f, 0.02539062f, 0.05258179f,
        0.49493408f, 0.42520142f, 0.02642822f, 0.05087280f, 0.49160767f, 0.42913818f, 0.02749634f,
        0.04922485f, 0.48822021f, 0.43307495f, 0.02859497f, 0.04760742f, 0.48477173f, 0.43695068f,
        0.02975464f, 0.04602051f, 0.48132324f, 0.44082642f, 0.03091431f, 0.04446411f, 0.47781372f,
        0.44467163f, 0.03210449f, 0.04293823f, 0.47424316f, 0.44845581f, 0.03335571f, 0.04147339f,
        0.47067261f, 0.45223999f, 0.03460693f, 0.04003906f, 0.46704102f, 0.45599365f, 0.03591919f,
        0.03863525f, 0.46340942f, 0.45971680f, 0.03726196f, 0.03726196f, 0.45971680f, 0.46340942f,
        0.03863525f, 0.03591919f, 0.45599365f, 0.46704102f, 0.04003906f, 0.03460693f, 0.45223999f,
        0.47067261f, 0.04147339f, 0.03335571f, 0.44845581f, 0.47424316f, 0.04293823f, 0.03210449f,
        0.44467163f, 0.47781372f, 0.04446411f, 0.03091431f, 0.44082642f, 0.48132324f, 0.04602051f,
        0.02975464f, 0.43695068f, 0.48477173f, 0.04760742f, 0.02859497f, 0.43307495f, 0.48822021f,
        0.04922485f, 0.02749634f, 0.42913818f, 0.49160767f, 0.05087280f, 0.02642822f, 0.42520142f,
        0.49493408f, 0.05258179f, 0.02539062f, 0.42123413f, 0.49826050f, 0.05432129f, 0.02435303f,
        0.41726685f, 0.50152588f, 0.05609131f, 0.02337646f, 0.41323853f, 0.50473022f, 0.05789185f,
        0.02243042f, 0.40921021f, 0.50790405f, 0.05975342f, 0.02148438f, 0.40518188f, 0.51101685f,
        0.06164551f, 0.02059937f, 0.40112305f, 0.51409912f, 0.06356812f, 0.01974487f, 0.39703369f,
        0.51715088f, 0.06552124f, 0.01889038f, 0.39294434f, 0.52014160f, 0.06753540f, 0.01806641f,
        0.38882446f, 0.52307129f, 0.06958008f, 0.01727295f, 0.38470459f, 0.52593994f, 0.07165527f,
        0.01651001f, 0.38055420f, 0.52877808f, 0.07379150f, 0.01577759f, 0.37640381f, 0.53155518f,
        0.07595825f, 0.01507568f, 0.37225342f, 0.53427124f, 0.07815552f, 0.01437378f, 0.36810303f,
        0.53695679f, 0.08041382f, 0.01370239f, 0.36392212f, 0.53958130f, 0.08267212f, 0.01306152f,
        0.35974121f, 0.54214478f, 0.08499146f, 0.01245117f, 0.35552979f, 0.54464722f, 0.08737183f,
        0.01184082f, 0.35134888f, 0.54708862f, 0.08978271f, 0.01126099f, 0.34713745f, 0.54949951f,
        0.09222412f, 0.01071167f, 0.34295654f, 0.55181885f, 0.09469604f, 0.01019287f, 0.33874512f,
        0.55410767f, 0.09722900f, 0.00967407f, 0.33453369f, 0.55633545f, 0.09976196f, 0.00915527f,
        0.33032227f, 0.55847168f, 0.10238647f, 0.00869751f, 0.32614136f, 0.56057739f, 0.10501099f,
        0.00823975f, 0.32192993f, 0.56262207f, 0.10769653f, 0.00778198f, 0.31771851f, 0.56457520f,
        0.11041260f, 0.00735474f, 0.31353760f, 0.56649780f, 0.11318970f, 0.00695801f, 0.30935669f,
        0.56835938f, 0.11596680f, 0.00656128f, 0.30517578f, 0.57012939f, 0.11880493f, 0.00616455f,
        0.30099487f, 0.57183838f, 0.12167358f, 0.00579834f, 0.29681396f, 0.57351685f, 0.12460327f,
        0.00546265f, 0.29266357f, 0.57510376f, 0.12753296f, 0.00512695f, 0.28851318f, 0.57662964f,
        0.13052368f, 0.00479126f, 0.28436279f, 0.57806396f, 0.13357544f, 0.00448608f, 0.28024292f,
        0.57946777f, 0.13662720f, 0.00418091f, 0.27612305f, 0.58078003f, 0.13973999f, 0.00387573f,
        0.27203369f, 0.58206177f, 0.14285278f, 0.00360107f, 0.26794434f, 0.58325195f, 0.14602661f,
        0.00332642f, 0.26385498f, 0.58435059f, 0.14923096f, 0.00308228f, 0.25979614f, 0.58541870f,
        0.15249634f, 0.00280762f, 0.25576782f, 0.58639526f, 0.15576172f, 0.00256348f, 0.25173950f,
        0.58731079f, 0.15908813f, 0.00234985f, 0.24774170f, 0.58816528f, 0.16244507f, 0.00210571f,
        0.24374390f, 0.58892822f, 0.16583252f, 0.00189209f, 0.23977661f, 0.58963013f, 0.16925049f,
        0.00167847f, 0.23583984f, 0.59027100f, 0.17269897f, 0.00146484f, 0.23193359f, 0.59085083f,
        0.17617798f, 0.00125122f, 0.22802734f, 0.59133911f, 0.17968750f, 0.00103760f, 0.22415161f,
        0.59176636f, 0.18322754f, 0.00085449f, 0.22030640f, 0.59213257f, 0.18679810f, 0.00067139f,
        0.21646118f, 0.59240723f, 0.19039917f, 0.00045776f, 0.21264648f, 0.59262085f, 0.19406128f,
        0.00027466f, 0.20889282f, 0.59277344f, 0.19772339f, 0.00009155f, 0.20513916f, 0.59283447f,
        0.20141602f,
    };

    static constexpr std::array<f32, 512> lut1 = {
        0.00207520f,  0.99606323f,  0.00210571f,  -0.00015259f, -0.00610352f, 0.99578857f,
        0.00646973f,  -0.00045776f, -0.01000977f, 0.99526978f,  0.01095581f,  -0.00079346f,
        -0.01373291f, 0.99444580f,  0.01562500f,  -0.00109863f, -0.01733398f, 0.99337769f,
        0.02041626f,  -0.00143433f, -0.02075195f, 0.99203491f,  0.02539062f,  -0.00177002f,
        -0.02404785f, 0.99041748f,  0.03051758f,  -0.00210571f, -0.02719116f, 0.98855591f,
        0.03582764f,  -0.00244141f, -0.03021240f, 0.98641968f,  0.04125977f,  -0.00280762f,
        -0.03308105f, 0.98400879f,  0.04687500f,  -0.00314331f, -0.03579712f, 0.98135376f,
        0.05261230f,  -0.00350952f, -0.03839111f, 0.97842407f,  0.05856323f,  -0.00390625f,
        -0.04083252f, 0.97521973f,  0.06463623f,  -0.00427246f, -0.04315186f, 0.97180176f,
        0.07086182f,  -0.00466919f, -0.04534912f, 0.96810913f,  0.07727051f,  -0.00509644f,
        -0.04742432f, 0.96414185f,  0.08383179f,  -0.00549316f, -0.04934692f, 0.95996094f,
        0.09054565f,  -0.00592041f, -0.05114746f, 0.95550537f,  0.09741211f,  -0.00637817f,
        -0.05285645f, 0.95083618f,  0.10443115f,  -0.00683594f, -0.05441284f, 0.94589233f,
        0.11160278f,  -0.00732422f, -0.05584717f, 0.94073486f,  0.11892700f,  -0.00781250f,
        -0.05718994f, 0.93533325f,  0.12643433f,  -0.00830078f, -0.05841064f, 0.92968750f,
        0.13406372f,  -0.00881958f, -0.05953979f, 0.92382812f,  0.14184570f,  -0.00936890f,
        -0.06054688f, 0.91772461f,  0.14978027f,  -0.00991821f, -0.06146240f, 0.91143799f,
        0.15783691f,  -0.01046753f, -0.06225586f, 0.90490723f,  0.16607666f,  -0.01104736f,
        -0.06295776f, 0.89816284f,  0.17443848f,  -0.01165771f, -0.06356812f, 0.89120483f,
        0.18292236f,  -0.01229858f, -0.06408691f, 0.88403320f,  0.19155884f,  -0.01293945f,
        -0.06451416f, 0.87667847f,  0.20034790f,  -0.01358032f, -0.06484985f, 0.86914062f,
        0.20925903f,  -0.01428223f, -0.06509399f, 0.86138916f,  0.21829224f,  -0.01495361f,
        -0.06527710f, 0.85345459f,  0.22744751f,  -0.01568604f, -0.06536865f, 0.84533691f,
        0.23675537f,  -0.01641846f, -0.06536865f, 0.83703613f,  0.24615479f,  -0.01718140f,
        -0.06533813f, 0.82858276f,  0.25567627f,  -0.01794434f, -0.06518555f, 0.81991577f,
        0.26531982f,  -0.01873779f, -0.06500244f, 0.81112671f,  0.27505493f,  -0.01956177f,
        -0.06472778f, 0.80215454f,  0.28491211f,  -0.02038574f, -0.06442261f, 0.79306030f,
        0.29489136f,  -0.02124023f, -0.06402588f, 0.78378296f,  0.30496216f,  -0.02209473f,
        -0.06359863f, 0.77438354f,  0.31512451f,  -0.02297974f, -0.06307983f, 0.76486206f,
        0.32537842f,  -0.02389526f, -0.06253052f, 0.75518799f,  0.33569336f,  -0.02481079f,
        -0.06195068f, 0.74539185f,  0.34613037f,  -0.02575684f, -0.06130981f, 0.73547363f,
        0.35662842f,  -0.02670288f, -0.06060791f, 0.72543335f,  0.36721802f,  -0.02767944f,
        -0.05987549f, 0.71527100f,  0.37786865f,  -0.02865601f, -0.05911255f, 0.70504761f,
        0.38858032f,  -0.02966309f, -0.05831909f, 0.69470215f,  0.39935303f,  -0.03067017f,
        -0.05746460f, 0.68426514f,  0.41018677f,  -0.03170776f, -0.05661011f, 0.67373657f,
        0.42108154f,  -0.03271484f, -0.05569458f, 0.66311646f,  0.43200684f,  -0.03378296f,
        -0.05477905f, 0.65246582f,  0.44299316f,  -0.03482056f, -0.05383301f, 0.64169312f,
        0.45401001f,  -0.03588867f, -0.05285645f, 0.63088989f,  0.46505737f,  -0.03695679f,
        -0.05187988f, 0.62002563f,  0.47613525f,  -0.03802490f, -0.05087280f, 0.60910034f,
        0.48721313f,  -0.03912354f, -0.04983521f, 0.59814453f,  0.49832153f,  -0.04019165f,
        -0.04879761f, 0.58712769f,  0.50946045f,  -0.04129028f, -0.04772949f, 0.57611084f,
        0.52056885f,  -0.04235840f, -0.04669189f, 0.56503296f,  0.53170776f,  -0.04345703f,
        -0.04562378f, 0.55392456f,  0.54281616f,  -0.04452515f, -0.04452515f, 0.54281616f,
        0.55392456f,  -0.04562378f, -0.04345703f, 0.53170776f,  0.56503296f,  -0.04669189f,
        -0.04235840f, 0.52056885f,  0.57611084f,  -0.04772949f, -0.04129028f, 0.50946045f,
        0.58712769f,  -0.04879761f, -0.04019165f, 0.49832153f,  0.59814453f,  -0.04983521f,
        -0.03912354f, 0.48721313f,  0.60910034f,  -0.05087280f, -0.03802490f, 0.47613525f,
        0.62002563f,  -0.05187988f, -0.03695679f, 0.46505737f,  0.63088989f,  -0.05285645f,
        -0.03588867f, 0.45401001f,  0.64169312f,  -0.05383301f, -0.03482056f, 0.44299316f,
        0.65246582f,  -0.05477905f, -0.03378296f, 0.43200684f,  0.66311646f,  -0.05569458f,
        -0.03271484f, 0.42108154f,  0.67373657f,  -0.05661011f, -0.03170776f, 0.41018677f,
        0.68426514f,  -0.05746460f, -0.03067017f, 0.39935303f,  0.69470215f,  -0.05831909f,
        -0.02966309f, 0.38858032f,  0.70504761f,  -0.05911255f, -0.02865601f, 0.37786865f,
        0.71527100f,  -0.05987549f, -0.02767944f, 0.36721802f,  0.72543335f,  -0.06060791f,
        -0.02670288f, 0.35662842f,  0.73547363f,  -0.06130981f, -0.02575684f, 0.34613037f,
        0.74539185f,  -0.06195068f, -0.02481079f, 0.33569336f,  0.75518799f,  -0.06253052f,
        -0.02389526f, 0.32537842f,  0.76486206f,  -0.06307983f, -0.02297974f, 0.31512451f,
        0.77438354f,  -0.06359863f, -0.02209473f, 0.30496216f,  0.78378296f,  -0.06402588f,
        -0.02124023f, 0.29489136f,  0.79306030f,  -0.06442261f, -0.02038574f, 0.28491211f,
        0.80215454f,  -0.06472778f, -0.01956177f, 0.27505493f,  0.81112671f,  -0.06500244f,
        -0.01873779f, 0.26531982f,  0.81991577f,  -0.06518555f, -0.01794434f, 0.25567627f,
        0.82858276f,  -0.06533813f, -0.01718140f, 0.24615479f,  0.83703613f,  -0.06536865f,
        -0.01641846f, 0.23675537f,  0.84533691f,  -0.06536865f, -0.01568604f, 0.22744751f,
        0.85345459f,  -0.06527710f, -0.01495361f, 0.21829224f,  0.86138916f,  -0.06509399f,
        -0.01428223f, 0.20925903f,  0.86914062f,  -0.06484985f, -0.01358032f, 0.20034790f,
        0.87667847f,  -0.06451416f, -0.01293945f, 0.19155884f,  0.88403320f,  -0.06408691f,
        -0.01229858f, 0.18292236f,  0.89120483f,  -0.06356812f, -0.01165771f, 0.17443848f,
        0.89816284f,  -0.06295776f, -0.01104736f, 0.16607666f,  0.90490723f,  -0.06225586f,
        -0.01046753f, 0.15783691f,  0.91143799f,  -0.06146240f, -0.00991821f, 0.14978027f,
        0.91772461f,  -0.06054688f, -0.00936890f, 0.14184570f,  0.92382812f,  -0.05953979f,
        -0.00881958f, 0.13406372f,  0.92968750f,  -0.05841064f, -0.00830078f, 0.12643433f,
        0.93533325f,  -0.05718994f, -0.00781250f, 0.11892700f,  0.94073486f,  -0.05584717f,
        -0.00732422f, 0.11160278f,  0.94589233f,  -0.05441284f, -0.00683594f, 0.10443115f,
        0.95083618f,  -0.05285645f, -0.00637817f, 0.09741211f,  0.95550537f,  -0.05114746f,
        -0.00592041f, 0.09054565f,  0.95996094f,  -0.04934692f, -0.00549316f, 0.08383179f,
        0.96414185f,  -0.04742432f, -0.00509644f, 0.07727051f,  0.96810913f,  -0.04534912f,
        -0.00466919f, 0.07086182f,  0.97180176f,  -0.04315186f, -0.00427246f, 0.06463623f,
        0.97521973f,  -0.04083252f, -0.00390625f, 0.05856323f,  0.97842407f,  -0.03839111f,
        -0.00350952f, 0.05261230f,  0.98135376f,  -0.03579712f, -0.00314331f, 0.04687500f,
        0.98400879f,  -0.03308105f, -0.00280762f, 0.04125977f,  0.98641968f,  -0.03021240f,
        -0.00244141f, 0.03582764f,  0.98855591f,  -0.02719116f, -0.00210571f, 0.03051758f,
        0.99041748f,  -0.02404785f, -0.00177002f, 0.02539062f,  0.99203491f,  -0.02075195f,
        -0.00143433f, 0.02041626f,  0.99337769f,  -0.01733398f, -0.00109863f, 0.01562500f,
        0.99444580f,  -0.01373291f, -0.00079346f, 0.01095581f,  0.99526978f,  -0.01000977f,
        -0.00045776f, 0.00646973f,  0.99578857f,  -0.00610352f, -0.00015259f, 0.00210571f,
        0.99606323f,  -0.00207520f,
    };

    static constexpr std::array<f32, 512> lut2 = {
        0.09750366f,  0.80221558f,  0.10159302f,  -0.00097656f, 0.09350586f,  0.80203247f,
        0.10580444f,  -0.00103760f, 0.08959961f,  0.80169678f,  0.11010742f,  -0.00115967f,
        0.08578491f,  0.80117798f,  0.11447144f,  -0.00128174f, 0.08203125f,  0.80047607f,
        0.11892700f,  -0.00140381f, 0.07836914f,  0.79962158f,  0.12347412f,  -0.00152588f,
        0.07479858f,  0.79861450f,  0.12814331f,  -0.00164795f, 0.07135010f,  0.79742432f,
        0.13287354f,  -0.00177002f, 0.06796265f,  0.79605103f,  0.13769531f,  -0.00192261f,
        0.06469727f,  0.79452515f,  0.14260864f,  -0.00204468f, 0.06149292f,  0.79284668f,
        0.14761353f,  -0.00219727f, 0.05834961f,  0.79098511f,  0.15270996f,  -0.00231934f,
        0.05532837f,  0.78894043f,  0.15789795f,  -0.00247192f, 0.05236816f,  0.78674316f,
        0.16317749f,  -0.00265503f, 0.04949951f,  0.78442383f,  0.16851807f,  -0.00280762f,
        0.04672241f,  0.78189087f,  0.17398071f,  -0.00299072f, 0.04400635f,  0.77920532f,
        0.17950439f,  -0.00314331f, 0.04141235f,  0.77636719f,  0.18511963f,  -0.00332642f,
        0.03887939f,  0.77337646f,  0.19082642f,  -0.00350952f, 0.03640747f,  0.77023315f,
        0.19659424f,  -0.00369263f, 0.03402710f,  0.76693726f,  0.20248413f,  -0.00387573f,
        0.03173828f,  0.76348877f,  0.20843506f,  -0.00405884f, 0.02951050f,  0.75985718f,
        0.21444702f,  -0.00427246f, 0.02737427f,  0.75610352f,  0.22055054f,  -0.00445557f,
        0.02529907f,  0.75219727f,  0.22674561f,  -0.00466919f, 0.02331543f,  0.74816895f,
        0.23300171f,  -0.00485229f, 0.02139282f,  0.74398804f,  0.23931885f,  -0.00506592f,
        0.01956177f,  0.73965454f,  0.24572754f,  -0.00531006f, 0.01779175f,  0.73519897f,
        0.25219727f,  -0.00552368f, 0.01605225f,  0.73059082f,  0.25872803f,  -0.00570679f,
        0.01440430f,  0.72586060f,  0.26535034f,  -0.00592041f, 0.01281738f,  0.72100830f,
        0.27203369f,  -0.00616455f, 0.01132202f,  0.71600342f,  0.27877808f,  -0.00637817f,
        0.00988770f,  0.71090698f,  0.28558350f,  -0.00656128f, 0.00851440f,  0.70565796f,
        0.29244995f,  -0.00677490f, 0.00720215f,  0.70031738f,  0.29934692f,  -0.00701904f,
        0.00592041f,  0.69485474f,  0.30633545f,  -0.00723267f, 0.00469971f,  0.68927002f,
        0.31338501f,  -0.00741577f, 0.00357056f,  0.68356323f,  0.32046509f,  -0.00762939f,
        0.00247192f,  0.67773438f,  0.32760620f,  -0.00787354f, 0.00143433f,  0.67184448f,
        0.33477783f,  -0.00808716f, 0.00045776f,  0.66583252f,  0.34197998f,  -0.00827026f,
        -0.00048828f, 0.65972900f,  0.34924316f,  -0.00845337f, -0.00134277f, 0.65353394f,
        0.35656738f,  -0.00863647f, -0.00216675f, 0.64721680f,  0.36389160f,  -0.00885010f,
        -0.00296021f, 0.64083862f,  0.37127686f,  -0.00903320f, -0.00369263f, 0.63433838f,
        0.37869263f,  -0.00921631f, -0.00436401f, 0.62777710f,  0.38613892f,  -0.00933838f,
        -0.00497437f, 0.62115479f,  0.39361572f,  -0.00949097f, -0.00558472f, 0.61444092f,
        0.40109253f,  -0.00964355f, -0.00613403f, 0.60763550f,  0.40859985f,  -0.00979614f,
        -0.00665283f, 0.60076904f,  0.41610718f,  -0.00991821f, -0.00714111f, 0.59384155f,
        0.42364502f,  -0.01000977f, -0.00756836f, 0.58685303f,  0.43121338f,  -0.01013184f,
        -0.00796509f, 0.57977295f,  0.43875122f,  -0.01022339f, -0.00833130f, 0.57266235f,
        0.44631958f,  -0.01028442f, -0.00866699f, 0.56552124f,  0.45388794f,  -0.01034546f,
        -0.00897217f, 0.55831909f,  0.46145630f,  -0.01040649f, -0.00921631f, 0.55105591f,
        0.46902466f,  -0.01040649f, -0.00946045f, 0.54373169f,  0.47659302f,  -0.01040649f,
        -0.00967407f, 0.53640747f,  0.48413086f,  -0.01037598f, -0.00985718f, 0.52902222f,
        0.49166870f,  -0.01037598f, -0.01000977f, 0.52160645f,  0.49917603f,  -0.01031494f,
        -0.01013184f, 0.51416016f,  0.50668335f,  -0.01025391f, -0.01025391f, 0.50668335f,
        0.51416016f,  -0.01013184f, -0.01031494f, 0.49917603f,  0.52160645f,  -0.01000977f,
        -0.01037598f, 0.49166870f,  0.52902222f,  -0.00985718f, -0.01037598f, 0.48413086f,
        0.53640747f,  -0.00967407f, -0.01040649f, 0.47659302f,  0.54373169f,  -0.00946045f,
        -0.01040649f, 0.46902466f,  0.55105591f,  -0.00921631f, -0.01040649f, 0.46145630f,
        0.55831909f,  -0.00897217f, -0.01034546f, 0.45388794f,  0.56552124f,  -0.00866699f,
        -0.01028442f, 0.44631958f,  0.57266235f,  -0.00833130f, -0.01022339f, 0.43875122f,
        0.57977295f,  -0.00796509f, -0.01013184f, 0.43121338f,  0.58685303f,  -0.00756836f,
        -0.01000977f, 0.42364502f,  0.59384155f,  -0.00714111f, -0.00991821f, 0.41610718f,
        0.60076904f,  -0.00665283f, -0.00979614f, 0.40859985f,  0.60763550f,  -0.00613403f,
        -0.00964355f, 0.40109253f,  0.61444092f,  -0.00558472f, -0.00949097f, 0.39361572f,
        0.62115479f,  -0.00497437f, -0.00933838f, 0.38613892f,  0.62777710f,  -0.00436401f,
        -0.00921631f, 0.37869263f,  0.63433838f,  -0.00369263f, -0.00903320f, 0.37127686f,
        0.64083862f,  -0.00296021f, -0.00885010f, 0.36389160f,  0.64721680f,  -0.00216675f,
        -0.00863647f, 0.35656738f,  0.65353394f,  -0.00134277f, -0.00845337f, 0.34924316f,
        0.65972900f,  -0.00048828f, -0.00827026f, 0.34197998f,  0.66583252f,  0.00045776f,
        -0.00808716f, 0.33477783f,  0.67184448f,  0.00143433f,  -0.00787354f, 0.32760620f,
        0.67773438f,  0.00247192f,  -0.00762939f, 0.32046509f,  0.68356323f,  0.00357056f,
        -0.00741577f, 0.31338501f,  0.68927002f,  0.00469971f,  -0.00723267f, 0.30633545f,
        0.69485474f,  0.00592041f,  -0.00701904f, 0.29934692f,  0.70031738f,  0.00720215f,
        -0.00677490f, 0.29244995f,  0.70565796f,  0.00851440f,  -0.00656128f, 0.28558350f,
        0.71090698f,  0.00988770f,  -0.00637817f, 0.27877808f,  0.71600342f,  0.01132202f,
        -0.00616455f, 0.27203369f,  0.72100830f,  0.01281738f,  -0.00592041f, 0.26535034f,
        0.72586060f,  0.01440430f,  -0.00570679f, 0.25872803f,  0.73059082f,  0.01605225f,
        -0.00552368f, 0.25219727f,  0.73519897f,  0.01779175f,  -0.00531006f, 0.24572754f,
        0.73965454f,  0.01956177f,  -0.00506592f, 0.23931885f,  0.74398804f,  0.02139282f,
        -0.00485229f, 0.23300171f,  0.74816895f,  0.02331543f,  -0.00466919f, 0.22674561f,
        0.75219727f,  0.02529907f,  -0.00445557f, 0.22055054f,  0.75610352f,  0.02737427f,
        -0.00427246f, 0.21444702f,  0.75985718f,  0.02951050f,  -0.00405884f, 0.20843506f,
        0.76348877f,  0.03173828f,  -0.00387573f, 0.20248413f,  0.76693726f,  0.03402710f,
        -0.00369263f, 0.19659424f,  0.77023315f,  0.03640747f,  -0.00350952f, 0.19082642f,
        0.77337646f,  0.03887939f,  -0.00332642f, 0.18511963f,  0.77636719f,  0.04141235f,
        -0.00314331f, 0.17950439f,  0.77920532f,  0.04400635f,  -0.00299072f, 0.17398071f,
        0.78189087f,  0.04672241f,  -0.00280762f, 0.16851807f,  0.78442383f,  0.04949951f,
        -0.00265503f, 0.16317749f,  0.78674316f,  0.05236816f,  -0.00247192f, 0.15789795f,
        0.78894043f,  0.05532837f,  -0.00231934f, 0.15270996f,  0.79098511f,  0.05834961f,
        -0.00219727f, 0.14761353f,  0.79284668f,  0.06149292f,  -0.00204468f, 0.14260864f,
        0.79452515f,  0.06469727f,  -0.00192261f, 0.13769531f,  0.79605103f,  0.06796265f,
        -0.00177002f, 0.13287354f,  0.79742432f,  0.07135010f,  -0.00164795f, 0.12814331f,
        0.79861450f,  0.07479858f,  -0.00152588f, 0.12347412f,  0.79962158f,  0.07836914f,
        -0.00140381f, 0.11892700f,  0.80047607f,  0.08203125f,  -0.00128174f, 0.11447144f,
        0.80117798f,  0.08578491f,  -0.00115967f, 0.11010742f,  0.80169678f,  0.08959961f,
        -0.00103760f, 0.10580444f,  0.80203247f,  0.09350586f,  -0.00097656f, 0.10159302f,
        0.80221558f,  0.09750366f,
    };

    const auto get_lut = [&]() -> std::span<const f32> {
        if (sample_rate_ratio <= 1.0f) {
            return std::span<const f32>(lut2.data(), lut2.size());
        } else if (sample_rate_ratio < 1.3f) {
            return std::span<const f32>(lut1.data(), lut1.size());
        } else {
            return std::span<const f32>(lut0.data(), lut0.size());
        }
    };

    auto lut{get_lut()};
    u32 read_index{0};
    for (u32 i = 0; i < samples_to_write; i++) {
        const auto lut_index{(fraction.get_frac() >> 8) * 4};
        const Common::FixedPoint<56, 8> sample0{input[read_index + 0] * lut[lut_index + 0]};
        const Common::FixedPoint<56, 8> sample1{input[read_index + 1] * lut[lut_index + 1]};
        const Common::FixedPoint<56, 8> sample2{input[read_index + 2] * lut[lut_index + 2]};
        const Common::FixedPoint<56, 8> sample3{input[read_index + 3] * lut[lut_index + 3]};
        output[i] = (sample0 + sample1 + sample2 + sample3).to_int_floor();
        fraction += sample_rate_ratio;
        read_index += static_cast<u32>(fraction.to_int_floor());
        fraction.clear_int();
    }
}

static void ResampleHighQuality(std::span<s32> output, std::span<const s16> input,
                                const Common::FixedPoint<49, 15>& sample_rate_ratio,
                                Common::FixedPoint<49, 15>& fraction, const u32 samples_to_write) {
    static constexpr std::array<f32, 1024> lut0 = {
        -0.01776123f, -0.00070190f, 0.26672363f,  0.50006104f,  0.26956177f,  0.00024414f,
        -0.01800537f, 0.00000000f,  -0.01748657f, -0.00164795f, 0.26388550f,  0.50003052f,
        0.27236938f,  0.00122070f,  -0.01824951f, -0.00003052f, -0.01724243f, -0.00256348f,
        0.26107788f,  0.49996948f,  0.27520752f,  0.00219727f,  -0.01849365f, -0.00003052f,
        -0.01699829f, -0.00344849f, 0.25823975f,  0.49984741f,  0.27801514f,  0.00320435f,
        -0.01873779f, -0.00006104f, -0.01675415f, -0.00433350f, 0.25543213f,  0.49972534f,
        0.28085327f,  0.00424194f,  -0.01898193f, -0.00006104f, -0.01651001f, -0.00518799f,
        0.25259399f,  0.49954224f,  0.28366089f,  0.00527954f,  -0.01922607f, -0.00009155f,
        -0.01626587f, -0.00604248f, 0.24978638f,  0.49932861f,  0.28646851f,  0.00634766f,
        -0.01947021f, -0.00012207f, -0.01602173f, -0.00686646f, 0.24697876f,  0.49908447f,
        0.28930664f,  0.00744629f,  -0.01971436f, -0.00015259f, -0.01574707f, -0.00765991f,
        0.24414062f,  0.49877930f,  0.29211426f,  0.00854492f,  -0.01995850f, -0.00015259f,
        -0.01550293f, -0.00845337f, 0.24133301f,  0.49847412f,  0.29492188f,  0.00967407f,
        -0.02020264f, -0.00018311f, -0.01525879f, -0.00921631f, 0.23852539f,  0.49810791f,
        0.29772949f,  0.01083374f,  -0.02044678f, -0.00021362f, -0.01501465f, -0.00997925f,
        0.23571777f,  0.49774170f,  0.30050659f,  0.01199341f,  -0.02069092f, -0.00024414f,
        -0.01477051f, -0.01071167f, 0.23291016f,  0.49731445f,  0.30331421f,  0.01318359f,
        -0.02093506f, -0.00027466f, -0.01452637f, -0.01141357f, 0.23010254f,  0.49685669f,
        0.30609131f,  0.01437378f,  -0.02117920f, -0.00030518f, -0.01428223f, -0.01211548f,
        0.22732544f,  0.49636841f,  0.30886841f,  0.01559448f,  -0.02142334f, -0.00033569f,
        -0.01403809f, -0.01278687f, 0.22451782f,  0.49581909f,  0.31164551f,  0.01684570f,
        -0.02163696f, -0.00039673f, -0.01379395f, -0.01345825f, 0.22174072f,  0.49526978f,
        0.31442261f,  0.01809692f,  -0.02188110f, -0.00042725f, -0.01358032f, -0.01409912f,
        0.21896362f,  0.49465942f,  0.31719971f,  0.01937866f,  -0.02209473f, -0.00045776f,
        -0.01333618f, -0.01473999f, 0.21618652f,  0.49404907f,  0.31994629f,  0.02069092f,
        -0.02233887f, -0.00048828f, -0.01309204f, -0.01535034f, 0.21343994f,  0.49337769f,
        0.32269287f,  0.02203369f,  -0.02255249f, -0.00054932f, -0.01284790f, -0.01596069f,
        0.21066284f,  0.49267578f,  0.32543945f,  0.02337646f,  -0.02279663f, -0.00057983f,
        -0.01263428f, -0.01654053f, 0.20791626f,  0.49194336f,  0.32818604f,  0.02471924f,
        -0.02301025f, -0.00064087f, -0.01239014f, -0.01708984f, 0.20516968f,  0.49118042f,
        0.33090210f,  0.02612305f,  -0.02322388f, -0.00067139f, -0.01214600f, -0.01763916f,
        0.20242310f,  0.49035645f,  0.33361816f,  0.02752686f,  -0.02343750f, -0.00073242f,
        -0.01193237f, -0.01818848f, 0.19970703f,  0.48953247f,  0.33633423f,  0.02896118f,
        -0.02365112f, -0.00079346f, -0.01168823f, -0.01867676f, 0.19696045f,  0.48864746f,
        0.33901978f,  0.03039551f,  -0.02386475f, -0.00082397f, -0.01147461f, -0.01919556f,
        0.19427490f,  0.48776245f,  0.34170532f,  0.03186035f,  -0.02407837f, -0.00088501f,
        -0.01123047f, -0.01968384f, 0.19155884f,  0.48681641f,  0.34439087f,  0.03335571f,
        -0.02429199f, -0.00094604f, -0.01101685f, -0.02014160f, 0.18887329f,  0.48583984f,
        0.34704590f,  0.03485107f,  -0.02447510f, -0.00100708f, -0.01080322f, -0.02059937f,
        0.18615723f,  0.48483276f,  0.34970093f,  0.03637695f,  -0.02468872f, -0.00106812f,
        -0.01058960f, -0.02102661f, 0.18350220f,  0.48379517f,  0.35235596f,  0.03793335f,
        -0.02487183f, -0.00112915f, -0.01034546f, -0.02145386f, 0.18081665f,  0.48272705f,
        0.35498047f,  0.03948975f,  -0.02505493f, -0.00119019f, -0.01013184f, -0.02188110f,
        0.17816162f,  0.48162842f,  0.35760498f,  0.04107666f,  -0.02523804f, -0.00125122f,
        -0.00991821f, -0.02227783f, 0.17550659f,  0.48049927f,  0.36019897f,  0.04269409f,
        -0.02542114f, -0.00131226f, -0.00970459f, -0.02264404f, 0.17288208f,  0.47933960f,
        0.36279297f,  0.04431152f,  -0.02560425f, -0.00140381f, -0.00952148f, -0.02301025f,
        0.17025757f,  0.47814941f,  0.36538696f,  0.04595947f,  -0.02578735f, -0.00146484f,
        -0.00930786f, -0.02337646f, 0.16763306f,  0.47689819f,  0.36795044f,  0.04763794f,
        -0.02593994f, -0.00152588f, -0.00909424f, -0.02371216f, 0.16503906f,  0.47564697f,
        0.37048340f,  0.04931641f,  -0.02609253f, -0.00161743f, -0.00888062f, -0.02401733f,
        0.16244507f,  0.47436523f,  0.37304688f,  0.05102539f,  -0.02627563f, -0.00170898f,
        -0.00869751f, -0.02435303f, 0.15988159f,  0.47302246f,  0.37554932f,  0.05276489f,
        -0.02642822f, -0.00177002f, -0.00848389f, -0.02462769f, 0.15731812f,  0.47167969f,
        0.37805176f,  0.05450439f,  -0.02658081f, -0.00186157f, -0.00830078f, -0.02493286f,
        0.15475464f,  0.47027588f,  0.38055420f,  0.05627441f,  -0.02670288f, -0.00195312f,
        -0.00808716f, -0.02520752f, 0.15222168f,  0.46887207f,  0.38302612f,  0.05804443f,
        -0.02685547f, -0.00204468f, -0.00790405f, -0.02545166f, 0.14968872f,  0.46743774f,
        0.38546753f,  0.05987549f,  -0.02697754f, -0.00213623f, -0.00772095f, -0.02569580f,
        0.14718628f,  0.46594238f,  0.38790894f,  0.06170654f,  -0.02709961f, -0.00222778f,
        -0.00753784f, -0.02593994f, 0.14468384f,  0.46444702f,  0.39031982f,  0.06353760f,
        -0.02722168f, -0.00231934f, -0.00735474f, -0.02615356f, 0.14218140f,  0.46289062f,
        0.39273071f,  0.06539917f,  -0.02734375f, -0.00241089f, -0.00717163f, -0.02636719f,
        0.13970947f,  0.46133423f,  0.39511108f,  0.06729126f,  -0.02743530f, -0.00250244f,
        -0.00698853f, -0.02655029f, 0.13726807f,  0.45974731f,  0.39749146f,  0.06918335f,
        -0.02755737f, -0.00259399f, -0.00680542f, -0.02673340f, 0.13479614f,  0.45812988f,
        0.39984131f,  0.07113647f,  -0.02764893f, -0.00271606f, -0.00662231f, -0.02691650f,
        0.13238525f,  0.45648193f,  0.40216064f,  0.07305908f,  -0.02774048f, -0.00280762f,
        -0.00643921f, -0.02706909f, 0.12997437f,  0.45480347f,  0.40447998f,  0.07504272f,
        -0.02780151f, -0.00292969f, -0.00628662f, -0.02722168f, 0.12756348f,  0.45309448f,
        0.40676880f,  0.07699585f,  -0.02789307f, -0.00305176f, -0.00610352f, -0.02734375f,
        0.12518311f,  0.45135498f,  0.40902710f,  0.07901001f,  -0.02795410f, -0.00314331f,
        -0.00595093f, -0.02746582f, 0.12280273f,  0.44958496f,  0.41128540f,  0.08102417f,
        -0.02801514f, -0.00326538f, -0.00579834f, -0.02758789f, 0.12045288f,  0.44778442f,
        0.41351318f,  0.08306885f,  -0.02804565f, -0.00338745f, -0.00561523f, -0.02770996f,
        0.11813354f,  0.44598389f,  0.41571045f,  0.08511353f,  -0.02810669f, -0.00350952f,
        -0.00546265f, -0.02780151f, 0.11581421f,  0.44412231f,  0.41787720f,  0.08718872f,
        -0.02813721f, -0.00363159f, -0.00531006f, -0.02786255f, 0.11349487f,  0.44226074f,
        0.42004395f,  0.08929443f,  -0.02816772f, -0.00375366f, -0.00515747f, -0.02795410f,
        0.11120605f,  0.44036865f,  0.42218018f,  0.09140015f,  -0.02816772f, -0.00387573f,
        -0.00500488f, -0.02801514f, 0.10894775f,  0.43844604f,  0.42431641f,  0.09353638f,
        -0.02819824f, -0.00402832f, -0.00485229f, -0.02807617f, 0.10668945f,  0.43649292f,
        0.42639160f,  0.09570312f,  -0.02819824f, -0.00415039f, -0.00469971f, -0.02810669f,
        0.10446167f,  0.43453979f,  0.42846680f,  0.09786987f,  -0.02819824f, -0.00427246f,
        -0.00457764f, -0.02813721f, 0.10223389f,  0.43252563f,  0.43051147f,  0.10003662f,
        -0.02816772f, -0.00442505f, -0.00442505f, -0.02816772f, 0.10003662f,  0.43051147f,
        0.43252563f,  0.10223389f,  -0.02813721f, -0.00457764f, -0.00427246f, -0.02819824f,
        0.09786987f,  0.42846680f,  0.43453979f,  0.10446167f,  -0.02810669f, -0.00469971f,
        -0.00415039f, -0.02819824f, 0.09570312f,  0.42639160f,  0.43649292f,  0.10668945f,
        -0.02807617f, -0.00485229f, -0.00402832f, -0.02819824f, 0.09353638f,  0.42431641f,
        0.43844604f,  0.10894775f,  -0.02801514f, -0.00500488f, -0.00387573f, -0.02816772f,
        0.09140015f,  0.42218018f,  0.44036865f,  0.11120605f,  -0.02795410f, -0.00515747f,
        -0.00375366f, -0.02816772f, 0.08929443f,  0.42004395f,  0.44226074f,  0.11349487f,
        -0.02786255f, -0.00531006f, -0.00363159f, -0.02813721f, 0.08718872f,  0.41787720f,
        0.44412231f,  0.11581421f,  -0.02780151f, -0.00546265f, -0.00350952f, -0.02810669f,
        0.08511353f,  0.41571045f,  0.44598389f,  0.11813354f,  -0.02770996f, -0.00561523f,
        -0.00338745f, -0.02804565f, 0.08306885f,  0.41351318f,  0.44778442f,  0.12045288f,
        -0.02758789f, -0.00579834f, -0.00326538f, -0.02801514f, 0.08102417f,  0.41128540f,
        0.44958496f,  0.12280273f,  -0.02746582f, -0.00595093f, -0.00314331f, -0.02795410f,
        0.07901001f,  0.40902710f,  0.45135498f,  0.12518311f,  -0.02734375f, -0.00610352f,
        -0.00305176f, -0.02789307f, 0.07699585f,  0.40676880f,  0.45309448f,  0.12756348f,
        -0.02722168f, -0.00628662f, -0.00292969f, -0.02780151f, 0.07504272f,  0.40447998f,
        0.45480347f,  0.12997437f,  -0.02706909f, -0.00643921f, -0.00280762f, -0.02774048f,
        0.07305908f,  0.40216064f,  0.45648193f,  0.13238525f,  -0.02691650f, -0.00662231f,
        -0.00271606f, -0.02764893f, 0.07113647f,  0.39984131f,  0.45812988f,  0.13479614f,
        -0.02673340f, -0.00680542f, -0.00259399f, -0.02755737f, 0.06918335f,  0.39749146f,
        0.45974731f,  0.13726807f,  -0.02655029f, -0.00698853f, -0.00250244f, -0.02743530f,
        0.06729126f,  0.39511108f,  0.46133423f,  0.13970947f,  -0.02636719f, -0.00717163f,
        -0.00241089f, -0.02734375f, 0.06539917f,  0.39273071f,  0.46289062f,  0.14218140f,
        -0.02615356f, -0.00735474f, -0.00231934f, -0.02722168f, 0.06353760f,  0.39031982f,
        0.46444702f,  0.14468384f,  -0.02593994f, -0.00753784f, -0.00222778f, -0.02709961f,
        0.06170654f,  0.38790894f,  0.46594238f,  0.14718628f,  -0.02569580f, -0.00772095f,
        -0.00213623f, -0.02697754f, 0.05987549f,  0.38546753f,  0.46743774f,  0.14968872f,
        -0.02545166f, -0.00790405f, -0.00204468f, -0.02685547f, 0.05804443f,  0.38302612f,
        0.46887207f,  0.15222168f,  -0.02520752f, -0.00808716f, -0.00195312f, -0.02670288f,
        0.05627441f,  0.38055420f,  0.47027588f,  0.15475464f,  -0.02493286f, -0.00830078f,
        -0.00186157f, -0.02658081f, 0.05450439f,  0.37805176f,  0.47167969f,  0.15731812f,
        -0.02462769f, -0.00848389f, -0.00177002f, -0.02642822f, 0.05276489f,  0.37554932f,
        0.47302246f,  0.15988159f,  -0.02435303f, -0.00869751f, -0.00170898f, -0.02627563f,
        0.05102539f,  0.37304688f,  0.47436523f,  0.16244507f,  -0.02401733f, -0.00888062f,
        -0.00161743f, -0.02609253f, 0.04931641f,  0.37048340f,  0.47564697f,  0.16503906f,
        -0.02371216f, -0.00909424f, -0.00152588f, -0.02593994f, 0.04763794f,  0.36795044f,
        0.47689819f,  0.16763306f,  -0.02337646f, -0.00930786f, -0.00146484f, -0.02578735f,
        0.04595947f,  0.36538696f,  0.47814941f,  0.17025757f,  -0.02301025f, -0.00952148f,
        -0.00140381f, -0.02560425f, 0.04431152f,  0.36279297f,  0.47933960f,  0.17288208f,
        -0.02264404f, -0.00970459f, -0.00131226f, -0.02542114f, 0.04269409f,  0.36019897f,
        0.48049927f,  0.17550659f,  -0.02227783f, -0.00991821f, -0.00125122f, -0.02523804f,
        0.04107666f,  0.35760498f,  0.48162842f,  0.17816162f,  -0.02188110f, -0.01013184f,
        -0.00119019f, -0.02505493f, 0.03948975f,  0.35498047f,  0.48272705f,  0.18081665f,
        -0.02145386f, -0.01034546f, -0.00112915f, -0.02487183f, 0.03793335f,  0.35235596f,
        0.48379517f,  0.18350220f,  -0.02102661f, -0.01058960f, -0.00106812f, -0.02468872f,
        0.03637695f,  0.34970093f,  0.48483276f,  0.18615723f,  -0.02059937f, -0.01080322f,
        -0.00100708f, -0.02447510f, 0.03485107f,  0.34704590f,  0.48583984f,  0.18887329f,
        -0.02014160f, -0.01101685f, -0.00094604f, -0.02429199f, 0.03335571f,  0.34439087f,
        0.48681641f,  0.19155884f,  -0.01968384f, -0.01123047f, -0.00088501f, -0.02407837f,
        0.03186035f,  0.34170532f,  0.48776245f,  0.19427490f,  -0.01919556f, -0.01147461f,
        -0.00082397f, -0.02386475f, 0.03039551f,  0.33901978f,  0.48864746f,  0.19696045f,
        -0.01867676f, -0.01168823f, -0.00079346f, -0.02365112f, 0.02896118f,  0.33633423f,
        0.48953247f,  0.19970703f,  -0.01818848f, -0.01193237f, -0.00073242f, -0.02343750f,
        0.02752686f,  0.33361816f,  0.49035645f,  0.20242310f,  -0.01763916f, -0.01214600f,
        -0.00067139f, -0.02322388f, 0.02612305f,  0.33090210f,  0.49118042f,  0.20516968f,
        -0.01708984f, -0.01239014f, -0.00064087f, -0.02301025f, 0.02471924f,  0.32818604f,
        0.49194336f,  0.20791626f,  -0.01654053f, -0.01263428f, -0.00057983f, -0.02279663f,
        0.02337646f,  0.32543945f,  0.49267578f,  0.21066284f,  -0.01596069f, -0.01284790f,
        -0.00054932f, -0.02255249f, 0.02203369f,  0.32269287f,  0.49337769f,  0.21343994f,
        -0.01535034f, -0.01309204f, -0.00048828f, -0.02233887f, 0.02069092f,  0.31994629f,
        0.49404907f,  0.21618652f,  -0.01473999f, -0.01333618f, -0.00045776f, -0.02209473f,
        0.01937866f,  0.31719971f,  0.49465942f,  0.21896362f,  -0.01409912f, -0.01358032f,
        -0.00042725f, -0.02188110f, 0.01809692f,  0.31442261f,  0.49526978f,  0.22174072f,
        -0.01345825f, -0.01379395f, -0.00039673f, -0.02163696f, 0.01684570f,  0.31164551f,
        0.49581909f,  0.22451782f,  -0.01278687f, -0.01403809f, -0.00033569f, -0.02142334f,
        0.01559448f,  0.30886841f,  0.49636841f,  0.22732544f,  -0.01211548f, -0.01428223f,
        -0.00030518f, -0.02117920f, 0.01437378f,  0.30609131f,  0.49685669f,  0.23010254f,
        -0.01141357f, -0.01452637f, -0.00027466f, -0.02093506f, 0.01318359f,  0.30331421f,
        0.49731445f,  0.23291016f,  -0.01071167f, -0.01477051f, -0.00024414f, -0.02069092f,
        0.01199341f,  0.30050659f,  0.49774170f,  0.23571777f,  -0.00997925f, -0.01501465f,
        -0.00021362f, -0.02044678f, 0.01083374f,  0.29772949f,  0.49810791f,  0.23852539f,
        -0.00921631f, -0.01525879f, -0.00018311f, -0.02020264f, 0.00967407f,  0.29492188f,
        0.49847412f,  0.24133301f,  -0.00845337f, -0.01550293f, -0.00015259f, -0.01995850f,
        0.00854492f,  0.29211426f,  0.49877930f,  0.24414062f,  -0.00765991f, -0.01574707f,
        -0.00015259f, -0.01971436f, 0.00744629f,  0.28930664f,  0.49908447f,  0.24697876f,
        -0.00686646f, -0.01602173f, -0.00012207f, -0.01947021f, 0.00634766f,  0.28646851f,
        0.49932861f,  0.24978638f,  -0.00604248f, -0.01626587f, -0.00009155f, -0.01922607f,
        0.00527954f,  0.28366089f,  0.49954224f,  0.25259399f,  -0.00518799f, -0.01651001f,
        -0.00006104f, -0.01898193f, 0.00424194f,  0.28085327f,  0.49972534f,  0.25543213f,
        -0.00433350f, -0.01675415f, -0.00006104f, -0.01873779f, 0.00320435f,  0.27801514f,
        0.49984741f,  0.25823975f,  -0.00344849f, -0.01699829f, -0.00003052f, -0.01849365f,
        0.00219727f,  0.27520752f,  0.49996948f,  0.26107788f,  -0.00256348f, -0.01724243f,
        -0.00003052f, -0.01824951f, 0.00122070f,  0.27236938f,  0.50003052f,  0.26388550f,
        -0.00164795f, -0.01748657f, 0.00000000f,  -0.01800537f, 0.00024414f,  0.26956177f,
        0.50006104f,  0.26672363f,  -0.00070190f, -0.01776123f,
    };

    static constexpr std::array<f32, 1024> lut1 = {
        0.01275635f,  -0.07745361f, 0.18670654f,  0.75119019f,  0.19219971f,  -0.07821655f,
        0.01272583f,  0.00000000f,  0.01281738f,  -0.07666016f, 0.18124390f,  0.75106812f,
        0.19772339f,  -0.07897949f, 0.01266479f,  0.00003052f,  0.01284790f,  -0.07583618f,
        0.17581177f,  0.75088501f,  0.20330811f,  -0.07971191f, 0.01257324f,  0.00006104f,
        0.01287842f,  -0.07501221f, 0.17044067f,  0.75057983f,  0.20892334f,  -0.08041382f,
        0.01248169f,  0.00009155f,  0.01290894f,  -0.07415771f, 0.16510010f,  0.75018311f,
        0.21453857f,  -0.08111572f, 0.01239014f,  0.00012207f,  0.01290894f,  -0.07330322f,
        0.15979004f,  0.74966431f,  0.22021484f,  -0.08178711f, 0.01229858f,  0.00015259f,
        0.01290894f,  -0.07241821f, 0.15454102f,  0.74908447f,  0.22592163f,  -0.08242798f,
        0.01217651f,  0.00018311f,  0.01290894f,  -0.07150269f, 0.14932251f,  0.74838257f,
        0.23165894f,  -0.08303833f, 0.01205444f,  0.00021362f,  0.01290894f,  -0.07058716f,
        0.14416504f,  0.74755859f,  0.23742676f,  -0.08364868f, 0.01193237f,  0.00024414f,
        0.01287842f,  -0.06967163f, 0.13903809f,  0.74667358f,  0.24322510f,  -0.08419800f,
        0.01177979f,  0.00027466f,  0.01284790f,  -0.06872559f, 0.13397217f,  0.74566650f,
        0.24905396f,  -0.08474731f, 0.01162720f,  0.00033569f,  0.01281738f,  -0.06777954f,
        0.12893677f,  0.74456787f,  0.25491333f,  -0.08526611f, 0.01147461f,  0.00036621f,
        0.01278687f,  -0.06683350f, 0.12396240f,  0.74337769f,  0.26077271f,  -0.08575439f,
        0.01129150f,  0.00042725f,  0.01275635f,  -0.06585693f, 0.11901855f,  0.74206543f,
        0.26669312f,  -0.08621216f, 0.01110840f,  0.00045776f,  0.01269531f,  -0.06488037f,
        0.11413574f,  0.74069214f,  0.27261353f,  -0.08663940f, 0.01092529f,  0.00051880f,
        0.01263428f,  -0.06387329f, 0.10931396f,  0.73919678f,  0.27853394f,  -0.08700562f,
        0.01071167f,  0.00057983f,  0.01257324f,  -0.06286621f, 0.10452271f,  0.73760986f,
        0.28451538f,  -0.08737183f, 0.01049805f,  0.00064087f,  0.01251221f,  -0.06185913f,
        0.09979248f,  0.73593140f,  0.29049683f,  -0.08770752f, 0.01025391f,  0.00067139f,
        0.01242065f,  -0.06082153f, 0.09512329f,  0.73413086f,  0.29647827f,  -0.08801270f,
        0.01000977f,  0.00073242f,  0.01232910f,  -0.05981445f, 0.09051514f,  0.73226929f,
        0.30249023f,  -0.08828735f, 0.00973511f,  0.00079346f,  0.01226807f,  -0.05877686f,
        0.08593750f,  0.73028564f,  0.30853271f,  -0.08850098f, 0.00949097f,  0.00088501f,
        0.01214600f,  -0.05773926f, 0.08142090f,  0.72824097f,  0.31457520f,  -0.08871460f,
        0.00918579f,  0.00094604f,  0.01205444f,  -0.05670166f, 0.07696533f,  0.72607422f,
        0.32061768f,  -0.08886719f, 0.00891113f,  0.00100708f,  0.01196289f,  -0.05563354f,
        0.07257080f,  0.72381592f,  0.32669067f,  -0.08898926f, 0.00860596f,  0.00106812f,
        0.01187134f,  -0.05459595f, 0.06820679f,  0.72146606f,  0.33276367f,  -0.08908081f,
        0.00827026f,  0.00115967f,  0.01174927f,  -0.05352783f, 0.06393433f,  0.71902466f,
        0.33883667f,  -0.08911133f, 0.00796509f,  0.00122070f,  0.01162720f,  -0.05245972f,
        0.05969238f,  0.71649170f,  0.34494019f,  -0.08914185f, 0.00759888f,  0.00131226f,
        0.01150513f,  -0.05139160f, 0.05551147f,  0.71389771f,  0.35101318f,  -0.08911133f,
        0.00726318f,  0.00137329f,  0.01138306f,  -0.05032349f, 0.05139160f,  0.71118164f,
        0.35711670f,  -0.08901978f, 0.00686646f,  0.00146484f,  0.01126099f,  -0.04928589f,
        0.04733276f,  0.70837402f,  0.36322021f,  -0.08892822f, 0.00650024f,  0.00155640f,
        0.01113892f,  -0.04821777f, 0.04333496f,  0.70550537f,  0.36932373f,  -0.08877563f,
        0.00610352f,  0.00164795f,  0.01101685f,  -0.04714966f, 0.03939819f,  0.70251465f,
        0.37542725f,  -0.08856201f, 0.00567627f,  0.00173950f,  0.01086426f,  -0.04608154f,
        0.03549194f,  0.69946289f,  0.38153076f,  -0.08834839f, 0.00527954f,  0.00183105f,
        0.01074219f,  -0.04501343f, 0.03167725f,  0.69631958f,  0.38763428f,  -0.08804321f,
        0.00482178f,  0.00192261f,  0.01058960f,  -0.04394531f, 0.02792358f,  0.69308472f,
        0.39370728f,  -0.08773804f, 0.00436401f,  0.00201416f,  0.01043701f,  -0.04287720f,
        0.02420044f,  0.68975830f,  0.39981079f,  -0.08737183f, 0.00390625f,  0.00210571f,
        0.01031494f,  -0.04180908f, 0.02056885f,  0.68637085f,  0.40588379f,  -0.08694458f,
        0.00344849f,  0.00222778f,  0.01016235f,  -0.04074097f, 0.01699829f,  0.68289185f,
        0.41195679f,  -0.08648682f, 0.00296021f,  0.00231934f,  0.01000977f,  -0.03970337f,
        0.01345825f,  0.67932129f,  0.41802979f,  -0.08596802f, 0.00244141f,  0.00244141f,
        0.00985718f,  -0.03863525f, 0.01000977f,  0.67568970f,  0.42407227f,  -0.08541870f,
        0.00192261f,  0.00253296f,  0.00970459f,  -0.03759766f, 0.00662231f,  0.67196655f,
        0.43011475f,  -0.08480835f, 0.00140381f,  0.00265503f,  0.00955200f,  -0.03652954f,
        0.00326538f,  0.66815186f,  0.43612671f,  -0.08416748f, 0.00085449f,  0.00277710f,
        0.00936890f,  -0.03549194f, 0.00000000f,  0.66427612f,  0.44213867f,  -0.08346558f,
        0.00027466f,  0.00289917f,  0.00921631f,  -0.03445435f, -0.00320435f, 0.66030884f,
        0.44812012f,  -0.08270264f, -0.00027466f, 0.00299072f,  0.00906372f,  -0.03344727f,
        -0.00634766f, 0.65631104f,  0.45407104f,  -0.08190918f, -0.00088501f, 0.00311279f,
        0.00891113f,  -0.03240967f, -0.00946045f, 0.65219116f,  0.46002197f,  -0.08105469f,
        -0.00146484f, 0.00323486f,  0.00872803f,  -0.03140259f, -0.01248169f, 0.64801025f,
        0.46594238f,  -0.08013916f, -0.00210571f, 0.00338745f,  0.00857544f,  -0.03039551f,
        -0.01544189f, 0.64376831f,  0.47183228f,  -0.07919312f, -0.00271606f, 0.00350952f,
        0.00842285f,  -0.02938843f, -0.01834106f, 0.63946533f,  0.47772217f,  -0.07818604f,
        -0.00335693f, 0.00363159f,  0.00823975f,  -0.02838135f, -0.02117920f, 0.63507080f,
        0.48358154f,  -0.07711792f, -0.00402832f, 0.00375366f,  0.00808716f,  -0.02740479f,
        -0.02395630f, 0.63061523f,  0.48937988f,  -0.07598877f, -0.00469971f, 0.00390625f,
        0.00793457f,  -0.02642822f, -0.02667236f, 0.62609863f,  0.49517822f,  -0.07482910f,
        -0.00537109f, 0.00402832f,  0.00775146f,  -0.02545166f, -0.02932739f, 0.62152100f,
        0.50094604f,  -0.07357788f, -0.00607300f, 0.00418091f,  0.00759888f,  -0.02450562f,
        -0.03192139f, 0.61685181f,  0.50665283f,  -0.07229614f, -0.00677490f, 0.00430298f,
        0.00741577f,  -0.02352905f, -0.03445435f, 0.61215210f,  0.51235962f,  -0.07098389f,
        -0.00750732f, 0.00445557f,  0.00726318f,  -0.02258301f, -0.03689575f, 0.60736084f,
        0.51800537f,  -0.06958008f, -0.00823975f, 0.00460815f,  0.00711060f,  -0.02166748f,
        -0.03930664f, 0.60253906f,  0.52362061f,  -0.06811523f, -0.00897217f, 0.00476074f,
        0.00692749f,  -0.02075195f, -0.04165649f, 0.59762573f,  0.52920532f,  -0.06661987f,
        -0.00973511f, 0.00488281f,  0.00677490f,  -0.01983643f, -0.04394531f, 0.59268188f,
        0.53475952f,  -0.06506348f, -0.01052856f, 0.00503540f,  0.00662231f,  -0.01892090f,
        -0.04617310f, 0.58767700f,  0.54025269f,  -0.06344604f, -0.01129150f, 0.00518799f,
        0.00643921f,  -0.01803589f, -0.04830933f, 0.58261108f,  0.54571533f,  -0.06173706f,
        -0.01208496f, 0.00534058f,  0.00628662f,  -0.01715088f, -0.05041504f, 0.57748413f,
        0.55111694f,  -0.05999756f, -0.01290894f, 0.00549316f,  0.00613403f,  -0.01626587f,
        -0.05245972f, 0.57232666f,  0.55648804f,  -0.05819702f, -0.01373291f, 0.00564575f,
        0.00598145f,  -0.01541138f, -0.05444336f, 0.56707764f,  0.56182861f,  -0.05636597f,
        -0.01455688f, 0.00582886f,  0.00582886f,  -0.01455688f, -0.05636597f, 0.56182861f,
        0.56707764f,  -0.05444336f, -0.01541138f, 0.00598145f,  0.00564575f,  -0.01373291f,
        -0.05819702f, 0.55648804f,  0.57232666f,  -0.05245972f, -0.01626587f, 0.00613403f,
        0.00549316f,  -0.01290894f, -0.05999756f, 0.55111694f,  0.57748413f,  -0.05041504f,
        -0.01715088f, 0.00628662f,  0.00534058f,  -0.01208496f, -0.06173706f, 0.54571533f,
        0.58261108f,  -0.04830933f, -0.01803589f, 0.00643921f,  0.00518799f,  -0.01129150f,
        -0.06344604f, 0.54025269f,  0.58767700f,  -0.04617310f, -0.01892090f, 0.00662231f,
        0.00503540f,  -0.01052856f, -0.06506348f, 0.53475952f,  0.59268188f,  -0.04394531f,
        -0.01983643f, 0.00677490f,  0.00488281f,  -0.00973511f, -0.06661987f, 0.52920532f,
        0.59762573f,  -0.04165649f, -0.02075195f, 0.00692749f,  0.00476074f,  -0.00897217f,
        -0.06811523f, 0.52362061f,  0.60253906f,  -0.03930664f, -0.02166748f, 0.00711060f,
        0.00460815f,  -0.00823975f, -0.06958008f, 0.51800537f,  0.60736084f,  -0.03689575f,
        -0.02258301f, 0.00726318f,  0.00445557f,  -0.00750732f, -0.07098389f, 0.51235962f,
        0.61215210f,  -0.03445435f, -0.02352905f, 0.00741577f,  0.00430298f,  -0.00677490f,
        -0.07229614f, 0.50665283f,  0.61685181f,  -0.03192139f, -0.02450562f, 0.00759888f,
        0.00418091f,  -0.00607300f, -0.07357788f, 0.50094604f,  0.62152100f,  -0.02932739f,
        -0.02545166f, 0.00775146f,  0.00402832f,  -0.00537109f, -0.07482910f, 0.49517822f,
        0.62609863f,  -0.02667236f, -0.02642822f, 0.00793457f,  0.00390625f,  -0.00469971f,
        -0.07598877f, 0.48937988f,  0.63061523f,  -0.02395630f, -0.02740479f, 0.00808716f,
        0.00375366f,  -0.00402832f, -0.07711792f, 0.48358154f,  0.63507080f,  -0.02117920f,
        -0.02838135f, 0.00823975f,  0.00363159f,  -0.00335693f, -0.07818604f, 0.47772217f,
        0.63946533f,  -0.01834106f, -0.02938843f, 0.00842285f,  0.00350952f,  -0.00271606f,
        -0.07919312f, 0.47183228f,  0.64376831f,  -0.01544189f, -0.03039551f, 0.00857544f,
        0.00338745f,  -0.00210571f, -0.08013916f, 0.46594238f,  0.64801025f,  -0.01248169f,
        -0.03140259f, 0.00872803f,  0.00323486f,  -0.00146484f, -0.08105469f, 0.46002197f,
        0.65219116f,  -0.00946045f, -0.03240967f, 0.00891113f,  0.00311279f,  -0.00088501f,
        -0.08190918f, 0.45407104f,  0.65631104f,  -0.00634766f, -0.03344727f, 0.00906372f,
        0.00299072f,  -0.00027466f, -0.08270264f, 0.44812012f,  0.66030884f,  -0.00320435f,
        -0.03445435f, 0.00921631f,  0.00289917f,  0.00027466f,  -0.08346558f, 0.44213867f,
        0.66427612f,  0.00000000f,  -0.03549194f, 0.00936890f,  0.00277710f,  0.00085449f,
        -0.08416748f, 0.43612671f,  0.66815186f,  0.00326538f,  -0.03652954f, 0.00955200f,
        0.00265503f,  0.00140381f,  -0.08480835f, 0.43011475f,  0.67196655f,  0.00662231f,
        -0.03759766f, 0.00970459f,  0.00253296f,  0.00192261f,  -0.08541870f, 0.42407227f,
        0.67568970f,  0.01000977f,  -0.03863525f, 0.00985718f,  0.00244141f,  0.00244141f,
        -0.08596802f, 0.41802979f,  0.67932129f,  0.01345825f,  -0.03970337f, 0.01000977f,
        0.00231934f,  0.00296021f,  -0.08648682f, 0.41195679f,  0.68289185f,  0.01699829f,
        -0.04074097f, 0.01016235f,  0.00222778f,  0.00344849f,  -0.08694458f, 0.40588379f,
        0.68637085f,  0.02056885f,  -0.04180908f, 0.01031494f,  0.00210571f,  0.00390625f,
        -0.08737183f, 0.39981079f,  0.68975830f,  0.02420044f,  -0.04287720f, 0.01043701f,
        0.00201416f,  0.00436401f,  -0.08773804f, 0.39370728f,  0.69308472f,  0.02792358f,
        -0.04394531f, 0.01058960f,  0.00192261f,  0.00482178f,  -0.08804321f, 0.38763428f,
        0.69631958f,  0.03167725f,  -0.04501343f, 0.01074219f,  0.00183105f,  0.00527954f,
        -0.08834839f, 0.38153076f,  0.69946289f,  0.03549194f,  -0.04608154f, 0.01086426f,
        0.00173950f,  0.00567627f,  -0.08856201f, 0.37542725f,  0.70251465f,  0.03939819f,
        -0.04714966f, 0.01101685f,  0.00164795f,  0.00610352f,  -0.08877563f, 0.36932373f,
        0.70550537f,  0.04333496f,  -0.04821777f, 0.01113892f,  0.00155640f,  0.00650024f,
        -0.08892822f, 0.36322021f,  0.70837402f,  0.04733276f,  -0.04928589f, 0.01126099f,
        0.00146484f,  0.00686646f,  -0.08901978f, 0.35711670f,  0.71118164f,  0.05139160f,
        -0.05032349f, 0.01138306f,  0.00137329f,  0.00726318f,  -0.08911133f, 0.35101318f,
        0.71389771f,  0.05551147f,  -0.05139160f, 0.01150513f,  0.00131226f,  0.00759888f,
        -0.08914185f, 0.34494019f,  0.71649170f,  0.05969238f,  -0.05245972f, 0.01162720f,
        0.00122070f,  0.00796509f,  -0.08911133f, 0.33883667f,  0.71902466f,  0.06393433f,
        -0.05352783f, 0.01174927f,  0.00115967f,  0.00827026f,  -0.08908081f, 0.33276367f,
        0.72146606f,  0.06820679f,  -0.05459595f, 0.01187134f,  0.00106812f,  0.00860596f,
        -0.08898926f, 0.32669067f,  0.72381592f,  0.07257080f,  -0.05563354f, 0.01196289f,
        0.00100708f,  0.00891113f,  -0.08886719f, 0.32061768f,  0.72607422f,  0.07696533f,
        -0.05670166f, 0.01205444f,  0.00094604f,  0.00918579f,  -0.08871460f, 0.31457520f,
        0.72824097f,  0.08142090f,  -0.05773926f, 0.01214600f,  0.00088501f,  0.00949097f,
        -0.08850098f, 0.30853271f,  0.73028564f,  0.08593750f,  -0.05877686f, 0.01226807f,
        0.00079346f,  0.00973511f,  -0.08828735f, 0.30249023f,  0.73226929f,  0.09051514f,
        -0.05981445f, 0.01232910f,  0.00073242f,  0.01000977f,  -0.08801270f, 0.29647827f,
        0.73413086f,  0.09512329f,  -0.06082153f, 0.01242065f,  0.00067139f,  0.01025391f,
        -0.08770752f, 0.29049683f,  0.73593140f,  0.09979248f,  -0.06185913f, 0.01251221f,
        0.00064087f,  0.01049805f,  -0.08737183f, 0.28451538f,  0.73760986f,  0.10452271f,
        -0.06286621f, 0.01257324f,  0.00057983f,  0.01071167f,  -0.08700562f, 0.27853394f,
        0.73919678f,  0.10931396f,  -0.06387329f, 0.01263428f,  0.00051880f,  0.01092529f,
        -0.08663940f, 0.27261353f,  0.74069214f,  0.11413574f,  -0.06488037f, 0.01269531f,
        0.00045776f,  0.01110840f,  -0.08621216f, 0.26669312f,  0.74206543f,  0.11901855f,
        -0.06585693f, 0.01275635f,  0.00042725f,  0.01129150f,  -0.08575439f, 0.26077271f,
        0.74337769f,  0.12396240f,  -0.06683350f, 0.01278687f,  0.00036621f,  0.01147461f,
        -0.08526611f, 0.25491333f,  0.74456787f,  0.12893677f,  -0.06777954f, 0.01281738f,
        0.00033569f,  0.01162720f,  -0.08474731f, 0.24905396f,  0.74566650f,  0.13397217f,
        -0.06872559f, 0.01284790f,  0.00027466f,  0.01177979f,  -0.08419800f, 0.24322510f,
        0.74667358f,  0.13903809f,  -0.06967163f, 0.01287842f,  0.00024414f,  0.01193237f,
        -0.08364868f, 0.23742676f,  0.74755859f,  0.14416504f,  -0.07058716f, 0.01290894f,
        0.00021362f,  0.01205444f,  -0.08303833f, 0.23165894f,  0.74838257f,  0.14932251f,
        -0.07150269f, 0.01290894f,  0.00018311f,  0.01217651f,  -0.08242798f, 0.22592163f,
        0.74908447f,  0.15454102f,  -0.07241821f, 0.01290894f,  0.00015259f,  0.01229858f,
        -0.08178711f, 0.22021484f,  0.74966431f,  0.15979004f,  -0.07330322f, 0.01290894f,
        0.00012207f,  0.01239014f,  -0.08111572f, 0.21453857f,  0.75018311f,  0.16510010f,
        -0.07415771f, 0.01290894f,  0.00009155f,  0.01248169f,  -0.08041382f, 0.20892334f,
        0.75057983f,  0.17044067f,  -0.07501221f, 0.01287842f,  0.00006104f,  0.01257324f,
        -0.07971191f, 0.20330811f,  0.75088501f,  0.17581177f,  -0.07583618f, 0.01284790f,
        0.00003052f,  0.01266479f,  -0.07897949f, 0.19772339f,  0.75106812f,  0.18124390f,
        -0.07666016f, 0.01281738f,  0.00000000f,  0.01272583f,  -0.07821655f, 0.19219971f,
        0.75119019f,  0.18670654f,  -0.07745361f, 0.01275635f,
    };

    static constexpr std::array<f32, 1024> lut2 = {
        -0.00036621f, 0.00143433f,  -0.00408936f, 0.99996948f,  0.00247192f,  -0.00048828f,
        0.00006104f,  0.00000000f,  -0.00079346f, 0.00329590f,  -0.01052856f, 0.99975586f,
        0.00918579f,  -0.00241089f, 0.00051880f,  -0.00003052f, -0.00122070f, 0.00512695f,
        -0.01684570f, 0.99929810f,  0.01605225f,  -0.00439453f, 0.00097656f,  -0.00006104f,
        -0.00161743f, 0.00689697f,  -0.02297974f, 0.99862671f,  0.02304077f,  -0.00640869f,
        0.00143433f,  -0.00009155f, -0.00201416f, 0.00866699f,  -0.02899170f, 0.99774170f,
        0.03018188f,  -0.00845337f, 0.00192261f,  -0.00015259f, -0.00238037f, 0.01037598f,
        -0.03488159f, 0.99664307f,  0.03741455f,  -0.01055908f, 0.00241089f,  -0.00018311f,
        -0.00274658f, 0.01202393f,  -0.04061890f, 0.99533081f,  0.04483032f,  -0.01266479f,
        0.00292969f,  -0.00024414f, -0.00308228f, 0.01364136f,  -0.04620361f, 0.99377441f,
        0.05233765f,  -0.01483154f, 0.00344849f,  -0.00027466f, -0.00341797f, 0.01522827f,
        -0.05163574f, 0.99200439f,  0.05999756f,  -0.01699829f, 0.00396729f,  -0.00033569f,
        -0.00375366f, 0.01678467f,  -0.05691528f, 0.99002075f,  0.06777954f,  -0.01922607f,
        0.00451660f,  -0.00039673f, -0.00405884f, 0.01828003f,  -0.06207275f, 0.98782349f,
        0.07568359f,  -0.02145386f, 0.00506592f,  -0.00042725f, -0.00436401f, 0.01971436f,
        -0.06707764f, 0.98541260f,  0.08370972f,  -0.02374268f, 0.00564575f,  -0.00048828f,
        -0.00463867f, 0.02114868f,  -0.07192993f, 0.98278809f,  0.09185791f,  -0.02603149f,
        0.00622559f,  -0.00054932f, -0.00494385f, 0.02252197f,  -0.07666016f, 0.97991943f,
        0.10012817f,  -0.02835083f, 0.00680542f,  -0.00061035f, -0.00518799f, 0.02383423f,
        -0.08123779f, 0.97686768f,  0.10848999f,  -0.03073120f, 0.00738525f,  -0.00070190f,
        -0.00543213f, 0.02511597f,  -0.08566284f, 0.97360229f,  0.11700439f,  -0.03308105f,
        0.00799561f,  -0.00076294f, -0.00567627f, 0.02636719f,  -0.08993530f, 0.97012329f,
        0.12561035f,  -0.03549194f, 0.00860596f,  -0.00082397f, -0.00592041f, 0.02755737f,
        -0.09405518f, 0.96643066f,  0.13436890f,  -0.03790283f, 0.00924683f,  -0.00091553f,
        -0.00613403f, 0.02868652f,  -0.09805298f, 0.96252441f,  0.14318848f,  -0.04034424f,
        0.00985718f,  -0.00097656f, -0.00631714f, 0.02981567f,  -0.10189819f, 0.95843506f,
        0.15213013f,  -0.04281616f, 0.01049805f,  -0.00106812f, -0.00653076f, 0.03085327f,
        -0.10559082f, 0.95413208f,  0.16119385f,  -0.04528809f, 0.01113892f,  -0.00112915f,
        -0.00671387f, 0.03189087f,  -0.10916138f, 0.94961548f,  0.17034912f,  -0.04779053f,
        0.01181030f,  -0.00122070f, -0.00686646f, 0.03286743f,  -0.11254883f, 0.94491577f,
        0.17959595f,  -0.05029297f, 0.01248169f,  -0.00131226f, -0.00701904f, 0.03378296f,
        -0.11584473f, 0.94000244f,  0.18893433f,  -0.05279541f, 0.01315308f,  -0.00140381f,
        -0.00717163f, 0.03466797f,  -0.11895752f, 0.93490601f,  0.19839478f,  -0.05532837f,
        0.01382446f,  -0.00149536f, -0.00732422f, 0.03552246f,  -0.12194824f, 0.92962646f,
        0.20791626f,  -0.05786133f, 0.01449585f,  -0.00158691f, -0.00744629f, 0.03631592f,
        -0.12478638f, 0.92413330f,  0.21752930f,  -0.06042480f, 0.01519775f,  -0.00167847f,
        -0.00753784f, 0.03707886f,  -0.12750244f, 0.91848755f,  0.22723389f,  -0.06298828f,
        0.01586914f,  -0.00177002f, -0.00765991f, 0.03781128f,  -0.13006592f, 0.91262817f,
        0.23703003f,  -0.06555176f, 0.01657104f,  -0.00189209f, -0.00775146f, 0.03848267f,
        -0.13250732f, 0.90658569f,  0.24691772f,  -0.06808472f, 0.01727295f,  -0.00198364f,
        -0.00784302f, 0.03909302f,  -0.13479614f, 0.90036011f,  0.25683594f,  -0.07064819f,
        0.01797485f,  -0.00210571f, -0.00790405f, 0.03970337f,  -0.13696289f, 0.89395142f,
        0.26687622f,  -0.07321167f, 0.01870728f,  -0.00219727f, -0.00796509f, 0.04025269f,
        -0.13900757f, 0.88739014f,  0.27694702f,  -0.07577515f, 0.01940918f,  -0.00231934f,
        -0.00802612f, 0.04077148f,  -0.14089966f, 0.88064575f,  0.28710938f,  -0.07833862f,
        0.02011108f,  -0.00244141f, -0.00808716f, 0.04122925f,  -0.14263916f, 0.87374878f,
        0.29733276f,  -0.08090210f, 0.02084351f,  -0.00253296f, -0.00811768f, 0.04165649f,
        -0.14428711f, 0.86666870f,  0.30761719f,  -0.08343506f, 0.02154541f,  -0.00265503f,
        -0.00814819f, 0.04205322f,  -0.14578247f, 0.85940552f,  0.31793213f,  -0.08596802f,
        0.02227783f,  -0.00277710f, -0.00814819f, 0.04238892f,  -0.14715576f, 0.85202026f,
        0.32833862f,  -0.08847046f, 0.02297974f,  -0.00289917f, -0.00817871f, 0.04272461f,
        -0.14840698f, 0.84445190f,  0.33874512f,  -0.09097290f, 0.02371216f,  -0.00302124f,
        -0.00817871f, 0.04299927f,  -0.14953613f, 0.83673096f,  0.34924316f,  -0.09347534f,
        0.02441406f,  -0.00314331f, -0.00817871f, 0.04321289f,  -0.15054321f, 0.82888794f,
        0.35977173f,  -0.09594727f, 0.02514648f,  -0.00326538f, -0.00814819f, 0.04342651f,
        -0.15142822f, 0.82086182f,  0.37033081f,  -0.09838867f, 0.02584839f,  -0.00341797f,
        -0.00814819f, 0.04357910f,  -0.15219116f, 0.81271362f,  0.38092041f,  -0.10079956f,
        0.02655029f,  -0.00354004f, -0.00811768f, 0.04373169f,  -0.15283203f, 0.80441284f,
        0.39154053f,  -0.10321045f, 0.02725220f,  -0.00366211f, -0.00808716f, 0.04382324f,
        -0.15338135f, 0.79598999f,  0.40219116f,  -0.10559082f, 0.02795410f,  -0.00381470f,
        -0.00805664f, 0.04388428f,  -0.15377808f, 0.78741455f,  0.41287231f,  -0.10794067f,
        0.02865601f,  -0.00393677f, -0.00799561f, 0.04388428f,  -0.15408325f, 0.77871704f,
        0.42358398f,  -0.11026001f, 0.02935791f,  -0.00405884f, -0.00793457f, 0.04388428f,
        -0.15426636f, 0.76989746f,  0.43429565f,  -0.11251831f, 0.03002930f,  -0.00421143f,
        -0.00787354f, 0.04385376f,  -0.15435791f, 0.76095581f,  0.44500732f,  -0.11477661f,
        0.03070068f,  -0.00433350f, -0.00781250f, 0.04379272f,  -0.15435791f, 0.75192261f,
        0.45574951f,  -0.11697388f, 0.03137207f,  -0.00448608f, -0.00775146f, 0.04367065f,
        -0.15420532f, 0.74273682f,  0.46649170f,  -0.11914062f, 0.03201294f,  -0.00460815f,
        -0.00769043f, 0.04354858f,  -0.15399170f, 0.73345947f,  0.47723389f,  -0.12127686f,
        0.03268433f,  -0.00473022f, -0.00759888f, 0.04339600f,  -0.15365601f, 0.72406006f,
        0.48794556f,  -0.12335205f, 0.03329468f,  -0.00488281f, -0.00750732f, 0.04321289f,
        -0.15322876f, 0.71456909f,  0.49868774f,  -0.12539673f, 0.03393555f,  -0.00500488f,
        -0.00741577f, 0.04296875f,  -0.15270996f, 0.70498657f,  0.50936890f,  -0.12738037f,
        0.03454590f,  -0.00515747f, -0.00732422f, 0.04272461f,  -0.15209961f, 0.69528198f,
        0.52008057f,  -0.12930298f, 0.03515625f,  -0.00527954f, -0.00723267f, 0.04248047f,
        -0.15136719f, 0.68551636f,  0.53076172f,  -0.13119507f, 0.03573608f,  -0.00543213f,
        -0.00714111f, 0.04217529f,  -0.15057373f, 0.67565918f,  0.54138184f,  -0.13299561f,
        0.03631592f,  -0.00555420f, -0.00701904f, 0.04183960f,  -0.14968872f, 0.66571045f,
        0.55200195f,  -0.13476562f, 0.03689575f,  -0.00567627f, -0.00692749f, 0.04150391f,
        -0.14871216f, 0.65567017f,  0.56259155f,  -0.13647461f, 0.03741455f,  -0.00582886f,
        -0.00680542f, 0.04113770f,  -0.14767456f, 0.64556885f,  0.57315063f,  -0.13812256f,
        0.03796387f,  -0.00595093f, -0.00668335f, 0.04074097f,  -0.14651489f, 0.63540649f,
        0.58364868f,  -0.13970947f, 0.03845215f,  -0.00607300f, -0.00656128f, 0.04031372f,
        -0.14529419f, 0.62518311f,  0.59411621f,  -0.14120483f, 0.03897095f,  -0.00619507f,
        -0.00643921f, 0.03988647f,  -0.14401245f, 0.61486816f,  0.60452271f,  -0.14263916f,
        0.03942871f,  -0.00631714f, -0.00631714f, 0.03942871f,  -0.14263916f, 0.60452271f,
        0.61486816f,  -0.14401245f, 0.03988647f,  -0.00643921f, -0.00619507f, 0.03897095f,
        -0.14120483f, 0.59411621f,  0.62518311f,  -0.14529419f, 0.04031372f,  -0.00656128f,
        -0.00607300f, 0.03845215f,  -0.13970947f, 0.58364868f,  0.63540649f,  -0.14651489f,
        0.04074097f,  -0.00668335f, -0.00595093f, 0.03796387f,  -0.13812256f, 0.57315063f,
        0.64556885f,  -0.14767456f, 0.04113770f,  -0.00680542f, -0.00582886f, 0.03741455f,
        -0.13647461f, 0.56259155f,  0.65567017f,  -0.14871216f, 0.04150391f,  -0.00692749f,
        -0.00567627f, 0.03689575f,  -0.13476562f, 0.55200195f,  0.66571045f,  -0.14968872f,
        0.04183960f,  -0.00701904f, -0.00555420f, 0.03631592f,  -0.13299561f, 0.54138184f,
        0.67565918f,  -0.15057373f, 0.04217529f,  -0.00714111f, -0.00543213f, 0.03573608f,
        -0.13119507f, 0.53076172f,  0.68551636f,  -0.15136719f, 0.04248047f,  -0.00723267f,
        -0.00527954f, 0.03515625f,  -0.12930298f, 0.52008057f,  0.69528198f,  -0.15209961f,
        0.04272461f,  -0.00732422f, -0.00515747f, 0.03454590f,  -0.12738037f, 0.50936890f,
        0.70498657f,  -0.15270996f, 0.04296875f,  -0.00741577f, -0.00500488f, 0.03393555f,
        -0.12539673f, 0.49868774f,  0.71456909f,  -0.15322876f, 0.04321289f,  -0.00750732f,
        -0.00488281f, 0.03329468f,  -0.12335205f, 0.48794556f,  0.72406006f,  -0.15365601f,
        0.04339600f,  -0.00759888f, -0.00473022f, 0.03268433f,  -0.12127686f, 0.47723389f,
        0.73345947f,  -0.15399170f, 0.04354858f,  -0.00769043f, -0.00460815f, 0.03201294f,
        -0.11914062f, 0.46649170f,  0.74273682f,  -0.15420532f, 0.04367065f,  -0.00775146f,
        -0.00448608f, 0.03137207f,  -0.11697388f, 0.45574951f,  0.75192261f,  -0.15435791f,
        0.04379272f,  -0.00781250f, -0.00433350f, 0.03070068f,  -0.11477661f, 0.44500732f,
        0.76095581f,  -0.15435791f, 0.04385376f,  -0.00787354f, -0.00421143f, 0.03002930f,
        -0.11251831f, 0.43429565f,  0.76989746f,  -0.15426636f, 0.04388428f,  -0.00793457f,
        -0.00405884f, 0.02935791f,  -0.11026001f, 0.42358398f,  0.77871704f,  -0.15408325f,
        0.04388428f,  -0.00799561f, -0.00393677f, 0.02865601f,  -0.10794067f, 0.41287231f,
        0.78741455f,  -0.15377808f, 0.04388428f,  -0.00805664f, -0.00381470f, 0.02795410f,
        -0.10559082f, 0.40219116f,  0.79598999f,  -0.15338135f, 0.04382324f,  -0.00808716f,
        -0.00366211f, 0.02725220f,  -0.10321045f, 0.39154053f,  0.80441284f,  -0.15283203f,
        0.04373169f,  -0.00811768f, -0.00354004f, 0.02655029f,  -0.10079956f, 0.38092041f,
        0.81271362f,  -0.15219116f, 0.04357910f,  -0.00814819f, -0.00341797f, 0.02584839f,
        -0.09838867f, 0.37033081f,  0.82086182f,  -0.15142822f, 0.04342651f,  -0.00814819f,
        -0.00326538f, 0.02514648f,  -0.09594727f, 0.35977173f,  0.82888794f,  -0.15054321f,
        0.04321289f,  -0.00817871f, -0.00314331f, 0.02441406f,  -0.09347534f, 0.34924316f,
        0.83673096f,  -0.14953613f, 0.04299927f,  -0.00817871f, -0.00302124f, 0.02371216f,
        -0.09097290f, 0.33874512f,  0.84445190f,  -0.14840698f, 0.04272461f,  -0.00817871f,
        -0.00289917f, 0.02297974f,  -0.08847046f, 0.32833862f,  0.85202026f,  -0.14715576f,
        0.04238892f,  -0.00814819f, -0.00277710f, 0.02227783f,  -0.08596802f, 0.31793213f,
        0.85940552f,  -0.14578247f, 0.04205322f,  -0.00814819f, -0.00265503f, 0.02154541f,
        -0.08343506f, 0.30761719f,  0.86666870f,  -0.14428711f, 0.04165649f,  -0.00811768f,
        -0.00253296f, 0.02084351f,  -0.08090210f, 0.29733276f,  0.87374878f,  -0.14263916f,
        0.04122925f,  -0.00808716f, -0.00244141f, 0.02011108f,  -0.07833862f, 0.28710938f,
        0.88064575f,  -0.14089966f, 0.04077148f,  -0.00802612f, -0.00231934f, 0.01940918f,
        -0.07577515f, 0.27694702f,  0.88739014f,  -0.13900757f, 0.04025269f,  -0.00796509f,
        -0.00219727f, 0.01870728f,  -0.07321167f, 0.26687622f,  0.89395142f,  -0.13696289f,
        0.03970337f,  -0.00790405f, -0.00210571f, 0.01797485f,  -0.07064819f, 0.25683594f,
        0.90036011f,  -0.13479614f, 0.03909302f,  -0.00784302f, -0.00198364f, 0.01727295f,
        -0.06808472f, 0.24691772f,  0.90658569f,  -0.13250732f, 0.03848267f,  -0.00775146f,
        -0.00189209f, 0.01657104f,  -0.06555176f, 0.23703003f,  0.91262817f,  -0.13006592f,
        0.03781128f,  -0.00765991f, -0.00177002f, 0.01586914f,  -0.06298828f, 0.22723389f,
        0.91848755f,  -0.12750244f, 0.03707886f,  -0.00753784f, -0.00167847f, 0.01519775f,
        -0.06042480f, 0.21752930f,  0.92413330f,  -0.12478638f, 0.03631592f,  -0.00744629f,
        -0.00158691f, 0.01449585f,  -0.05786133f, 0.20791626f,  0.92962646f,  -0.12194824f,
        0.03552246f,  -0.00732422f, -0.00149536f, 0.01382446f,  -0.05532837f, 0.19839478f,
        0.93490601f,  -0.11895752f, 0.03466797f,  -0.00717163f, -0.00140381f, 0.01315308f,
        -0.05279541f, 0.18893433f,  0.94000244f,  -0.11584473f, 0.03378296f,  -0.00701904f,
        -0.00131226f, 0.01248169f,  -0.05029297f, 0.17959595f,  0.94491577f,  -0.11254883f,
        0.03286743f,  -0.00686646f, -0.00122070f, 0.01181030f,  -0.04779053f, 0.17034912f,
        0.94961548f,  -0.10916138f, 0.03189087f,  -0.00671387f, -0.00112915f, 0.01113892f,
        -0.04528809f, 0.16119385f,  0.95413208f,  -0.10559082f, 0.03085327f,  -0.00653076f,
        -0.00106812f, 0.01049805f,  -0.04281616f, 0.15213013f,  0.95843506f,  -0.10189819f,
        0.02981567f,  -0.00631714f, -0.00097656f, 0.00985718f,  -0.04034424f, 0.14318848f,
        0.96252441f,  -0.09805298f, 0.02868652f,  -0.00613403f, -0.00091553f, 0.00924683f,
        -0.03790283f, 0.13436890f,  0.96643066f,  -0.09405518f, 0.02755737f,  -0.00592041f,
        -0.00082397f, 0.00860596f,  -0.03549194f, 0.12561035f,  0.97012329f,  -0.08993530f,
        0.02636719f,  -0.00567627f, -0.00076294f, 0.00799561f,  -0.03308105f, 0.11700439f,
        0.97360229f,  -0.08566284f, 0.02511597f,  -0.00543213f, -0.00070190f, 0.00738525f,
        -0.03073120f, 0.10848999f,  0.97686768f,  -0.08123779f, 0.02383423f,  -0.00518799f,
        -0.00061035f, 0.00680542f,  -0.02835083f, 0.10012817f,  0.97991943f,  -0.07666016f,
        0.02252197f,  -0.00494385f, -0.00054932f, 0.00622559f,  -0.02603149f, 0.09185791f,
        0.98278809f,  -0.07192993f, 0.02114868f,  -0.00463867f, -0.00048828f, 0.00564575f,
        -0.02374268f, 0.08370972f,  0.98541260f,  -0.06707764f, 0.01971436f,  -0.00436401f,
        -0.00042725f, 0.00506592f,  -0.02145386f, 0.07568359f,  0.98782349f,  -0.06207275f,
        0.01828003f,  -0.00405884f, -0.00039673f, 0.00451660f,  -0.01922607f, 0.06777954f,
        0.99002075f,  -0.05691528f, 0.01678467f,  -0.00375366f, -0.00033569f, 0.00396729f,
        -0.01699829f, 0.05999756f,  0.99200439f,  -0.05163574f, 0.01522827f,  -0.00341797f,
        -0.00027466f, 0.00344849f,  -0.01483154f, 0.05233765f,  0.99377441f,  -0.04620361f,
        0.01364136f,  -0.00308228f, -0.00024414f, 0.00292969f,  -0.01266479f, 0.04483032f,
        0.99533081f,  -0.04061890f, 0.01202393f,  -0.00274658f, -0.00018311f, 0.00241089f,
        -0.01055908f, 0.03741455f,  0.99664307f,  -0.03488159f, 0.01037598f,  -0.00238037f,
        -0.00015259f, 0.00192261f,  -0.00845337f, 0.03018188f,  0.99774170f,  -0.02899170f,
        0.00866699f,  -0.00201416f, -0.00009155f, 0.00143433f,  -0.00640869f, 0.02304077f,
        0.99862671f,  -0.02297974f, 0.00689697f,  -0.00161743f, -0.00006104f, 0.00097656f,
        -0.00439453f, 0.01605225f,  0.99929810f,  -0.01684570f, 0.00512695f,  -0.00122070f,
        -0.00003052f, 0.00051880f,  -0.00241089f, 0.00918579f,  0.99975586f,  -0.01052856f,
        0.00329590f,  -0.00079346f, 0.00000000f,  0.00006104f,  -0.00048828f, 0.00247192f,
        0.99996948f,  -0.00408936f, 0.00143433f,  -0.00036621f,
    };

    const auto get_lut = [&]() -> std::span<const f32> {
        if (sample_rate_ratio <= 1.0f) {
            return std::span<const f32>(lut2.data(), lut2.size());
        } else if (sample_rate_ratio < 1.3f) {
            return std::span<const f32>(lut1.data(), lut1.size());
        } else {
            return std::span<const f32>(lut0.data(), lut0.size());
        }
    };

    auto lut{get_lut()};
    u32 read_index{0};
    for (u32 i = 0; i < samples_to_write; i++) {
        const auto lut_index{(fraction.get_frac() >> 8) * 8};
        const Common::FixedPoint<56, 8> sample0{input[read_index + 0] * lut[lut_index + 0]};
        const Common::FixedPoint<56, 8> sample1{input[read_index + 1] * lut[lut_index + 1]};
        const Common::FixedPoint<56, 8> sample2{input[read_index + 2] * lut[lut_index + 2]};
        const Common::FixedPoint<56, 8> sample3{input[read_index + 3] * lut[lut_index + 3]};
        const Common::FixedPoint<56, 8> sample4{input[read_index + 4] * lut[lut_index + 4]};
        const Common::FixedPoint<56, 8> sample5{input[read_index + 5] * lut[lut_index + 5]};
        const Common::FixedPoint<56, 8> sample6{input[read_index + 6] * lut[lut_index + 6]};
        const Common::FixedPoint<56, 8> sample7{input[read_index + 7] * lut[lut_index + 7]};
        output[i] = (sample0 + sample1 + sample2 + sample3 + sample4 + sample5 + sample6 + sample7)
                        .to_int_floor();
        fraction += sample_rate_ratio;
        read_index += static_cast<u32>(fraction.to_int_floor());
        fraction.clear_int();
    }
}

void Resample(std::span<s32> output, std::span<const s16> input,
              const Common::FixedPoint<49, 15>& sample_rate_ratio,
              Common::FixedPoint<49, 15>& fraction, const u32 samples_to_write,
              const SrcQuality src_quality) {

    switch (src_quality) {
    case SrcQuality::Low:
        ResampleLowQuality(output, input, sample_rate_ratio, fraction, samples_to_write);
        break;
    case SrcQuality::Medium:
        ResampleNormalQuality(output, input, sample_rate_ratio, fraction, samples_to_write);
        break;
    case SrcQuality::High:
        ResampleHighQuality(output, input, sample_rate_ratio, fraction, samples_to_write);
        break;
    }
}

} // namespace AudioCore::Renderer
