/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/negative-binomial/cdf' );
import NegativeBinomial = require( './../../../../../base/dists/negative-binomial/ctor' );
import kurtosis = require( './../../../../../base/dists/negative-binomial/kurtosis' );
import logpmf = require( './../../../../../base/dists/negative-binomial/logpmf' );
import mean = require( './../../../../../base/dists/negative-binomial/mean' );
import mgf = require( './../../../../../base/dists/negative-binomial/mgf' );
import mode = require( './../../../../../base/dists/negative-binomial/mode' );
import pmf = require( './../../../../../base/dists/negative-binomial/pmf' );
import quantile = require( './../../../../../base/dists/negative-binomial/quantile' );
import skewness = require( './../../../../../base/dists/negative-binomial/skewness' );
import stdev = require( './../../../../../base/dists/negative-binomial/stdev' );
import variance = require( './../../../../../base/dists/negative-binomial/variance' );

/**
* Interface describing the `negative-binomial` namespace.
*/
interface Namespace {
	/**
	* Negative binomial distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param r - number of successes until experiment is stopped
	* @param p - success probability
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 5.0, 20.0, 0.8 );
	* // returns ~0.617
	*
	* y = ns.cdf( 21.0, 20.0, 0.5 );
	* // returns ~0.622
	*
	* y = ns.cdf( 5.0, 10.0, 0.4 );
	* // returns ~0.034
	*
	* y = ns.cdf( 0.0, 10.0, 0.9 );
	* // returns ~0.349
	*
	* y = ns.cdf( 21.0, 15.5, 0.5 );
	* // returns ~0.859
	*
	* y = ns.cdf( 5.0, 7.4, 0.4 );
	* // returns ~0.131
	*
	* var mycdf = ns.cdf.factory( 10, 0.5 );
	* y = mycdf( 3.0 );
	* // returns ~0.046
	*
	* y = mycdf( 11.0 );
	* // returns ~0.668
	*/
	cdf: typeof cdf;

	/**
	* Negative Binomial Distribution.
	*/
	NegativeBinomial: typeof NegativeBinomial;

	/**
	* Returns the excess kurtosis of a negative binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a `r` which is not a positive number, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param r - number of failures until experiment is stopped
	* @param p - success probability
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 100, 0.2 );
	* // returns ~0.061
	*
	* @example
	* var v = ns.kurtosis( 20, 0.5 );
	* // returns ~0.325
	*
	* @example
	* var v = ns.kurtosis( 10.3, 0.8 );
	* // returns ~0.893
	*
	* @example
	* var v = ns.kurtosis( -2, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 20, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 0.5 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Negative binomial distribution natural logarithm of the probability mass function (logPMF).
	*
	* @param x - input value
	* @param r - number of successes until experiment is stopped
	* @param p - success probability
	* @returns evaluated logPMF
	*
	* @example
	* var y = ns.logpmf( 3.0, 20, 0.2 );
	* // returns ~-1.583
	*
	* y = ns.logpmf( 21.0, 20, 0.2 );
	* // returns -Infinity
	*
	* y = ns.logpmf( 5.0, 10, 0.4 );
	* // returns ~-1.606
	*
	* y = ns.logpmf( 0.0, 10, 0.4 );
	* // returns ~-5.108
	*
	* var mylogpmf = ns.logpmf.factory( 10, 0.5 );
	*
	* y = mylogpmf( 3.0 );
	* // returns ~-2.146
	*
	* y = mylogpmf( 5.0 );
	* // returns ~-1.402
	*/
	logpmf: typeof logpmf;

	/**
	* Returns the expected value of a negative binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a `r` which is not a positive number, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param r - number of failures until experiment is stopped
	* @param p - success probability
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 100, 0.2 );
	* // returns 400.0
	*
	* @example
	* var v = ns.mean( 20, 0.5 );
	* // returns 20.0
	*
	* @example
	* var v = ns.mean( 10.3, 0.8 );
	* // returns ~2.575
	*
	* @example
	* var v = ns.mean( -2, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 20, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 0.5 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Negative binomial distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param r - number of successes until experiment is stopped
	* @param p - success probability
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.05, 20.0, 0.8 );
	* // returns ~267.839
	*
	* y = ns.mgf( 0.1, 20.0, 0.1 );
	* // returns ~9.347
	*
	* y = ns.mgf( 0.5, 10.0, 0.4 );
	* // returns ~42822.023
	*
	* var myMGF = ns.mgf.factory( 4.3, 0.4 );
	* y = myMGF( 0.2 );
	* // returns ~4.696
	*
	* y = myMGF( 0.4 );
	* // returns ~30.83
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a negative binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a `r` which is not a positive number, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param r - number of failures until experiment is stopped
	* @param p - success probability
	* @returns mode
	*
	* @example
	* var v = ns.mode( 100, 0.2 );
	* // returns 396
	*
	* @example
	* var v = ns.mode( 20, 0.5 );
	* // returns 19
	*
	* @example
	* var v = ns.mode( 10.3, 0.8 );
	* // returns 2
	*
	* @example
	* var v = ns.mode( -2, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 20, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 0.5 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Negative binomial distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param r - number of successes until experiment is stopped
	* @param p - success probability
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 5.0, 20.0, 0.8 );
	* // returns ~0.157
	*
	* y = ns.pmf( 21.0, 20.0, 0.5 );
	* // returns ~0.06
	*
	* y = ns.pmf( 5.0, 10.0, 0.4 );
	* // returns ~0.016
	*
	* y = ns.pmf( 0.0, 10.0, 0.9 );
	* // returns ~0.349
	*
	* y = ns.pmf( 21.0, 15.5, 0.5 );
	* // returns ~0.037
	*
	* y = ns.pmf( 5.0, 7.4, 0.4 );
	* // returns ~0.051
	*
	* var mypmf = ns.pmf.factory( 10, 0.5 );
	* y = mypmf( 3.0 );
	* // returns ~0.027
	*
	* y = mypmf( 5.0 );
	* // returns ~0.061
	*/
	pmf: typeof pmf;

	/**
	* Negative binomial distribution quantile function.
	*
	* @param k - input value
	* @param r - number of successes until experiment is stopped
	* @param p - success probability
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.9, 20.0, 0.2 );
	* // returns 106
	*
	* y = ns.quantile( 0.9, 20.0, 0.8 );
	* // returns 8
	*
	* var myquantile = ns.quantile.factory( 10.0, 0.5 );
	* y = myquantile( 0.1 );
	* // returns 5
	*
	* y = myquantile( 0.9 );
	* // returns 16
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a negative binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a `r` which is not a positive number, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param r - number of failures until experiment is stopped
	* @param p - success probability
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 100, 0.2 );
	* // returns ~0.201
	*
	* @example
	* var v = ns.skewness( 20, 0.5 );
	* // returns ~0.474
	*
	* @example
	* var v = ns.skewness( 10.3, 0.8 );
	* // returns ~0.836
	*
	* @example
	* var v = ns.skewness( -2, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 20, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 0.5 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a negative binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a `r` which is not a positive number, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param r - number of failures until experiment is stopped
	* @param p - success probability
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 100, 0.2 );
	* // returns ~44.721
	*
	* @example
	* var v = ns.stdev( 20, 0.5 );
	* // returns ~6.325
	*
	* @example
	* var v = ns.stdev( 10.3, 0.8 );
	* // returns ~1.794
	*
	* @example
	* var v = ns.stdev( -2, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 20, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 0.5 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a negative binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a `r` which is not a positive number, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param r - number of failures until experiment is stopped
	* @param p - success probability
	* @returns variance
	*
	* @example
	* var v = ns.variance( 100, 0.2 );
	* // returns 2000.0
	*
	* @example
	* var v = ns.variance( 20, 0.5 );
	* // returns 40.0
	*
	* @example
	* var v = ns.variance( 10.3, 0.8 );
	* // returns ~3.219
	*
	* @example
	* var v = ns.variance( -2, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 20, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 0.5 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Negative binomial distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
